//
// Mech Valentina - Your "Low T" Auto-Co-Pilot.
//
// usage: 
// place MechVal files in Ships/Script
// Open your fav text editor on MechValCmd.ks
// open a kOS Terminal from your vessel 
// type: switch to 0. <enter>
// type: run MechValCmd.
//       any script in MechValCmd.ks will be executed

// remove all nodes
DECLARE FUNCTION mvClearNodes {
	UNTIL NOT HASNODE {
		REMOVE NEXTNODE.
		WAIT 0.
	}.
}.

// eta utSeconds :r:n:p period
DECLARE FUNCTION mvNodeDesc {
	DECLARE PARAMETER N. // node
	RETURN ""+mvSecondsStr( ROUND(N:ETA) )+" "+( TIME:SECONDS+N:ETA )
		+" :"+N:RADIALOUT+":"+N:NORMAL+":"+N:PROGRADE+" "
		+mvSecondsStr( ROUND(N:ORBIT:PERIOD) ).
}.

// print all nodes on the terminal
DECLARE FUNCTION mvPrintNodes {
	local x is 1.
	FOR ANM IN ALLNODES {
		PRINT "Node."+x+": "+mvNodeDesc( ANM ).
		set x to x+1.
	}.
}.

// change the magnitude of a node
DECLARE FUNCTION mvDeltaMagNode {
	DECLARE PARAMETER ni. // starting with 1
	DECLARE PARAMETER dM.

	SET N to ALLNODES[ni-1].
	SET NVec to mvNodeVec(N).
	SET dvn to dM * ( NVec:NORMALIZED ).
	mvVecNode( N, NVec+dvn ).
}

// change the magnitude of a node, taking the delta from another node
DECLARE FUNCTION mvDeltaMagNodeFrom {
	DECLARE PARAMETER ni. // target node, starting with 1
	DECLARE PARAMETER di. // donor node, starting with 1
	DECLARE PARAMETER dM.

	SET N to ALLNODES[ni-1].
	SET D to ALLNODES[di-1].
	SET NVec to mvNodeVec(N).
	SET dvn to dM * ( NVec:NORMALIZED ).
	mvVecNode( N, NVec+dvn ).
	mvVecNode( D, mvNodeVec(D)-dvn ).
}

// turn a node into a (n,r,p) vector representation
// this is different to the burn vector
DECLARE FUNCTION mvNodeVec {
	DECLARE PARAMETER n.
	RETURN V( n:RADIALOUT, n:NORMAL, n:PROGRADE ).
}.

// set a node from (n,r,p) vector
DECLARE FUNCTION mvVecNode {
	DECLARE PARAMETER n. // target node
	DECLARE PARAMETER v. // vector to apply
	SET n:RADIALOUT to v:X.
	SET n:NORMAL to v:Y.
	SET n:PROGRADE to v:Z.
}.

// split a node into two
DECLARE FUNCTION mvSplitNode {
	DECLARE PARAMETER N. // node number startin at 1
	DECLARE PARAMETER M. // magnitude to leave in first node

	Set sNode to ALLNODES[N-1].
	Set newNode to NODE( TIME:SECONDS + 1 + sNode:ETA, sNode:RADIALOUT, sNode:NORMAL, sNode:PROGRADE - M ).
	ADD newNode.
	set sNode:PROGRADE to M.
}.

// node split into magnitude chunks using current throttle setting
DECLARE FUNCTION mvSplitsNodeMag{
	DECLARE PARAMETER N. // node number from 1
	DECLARE PARAMETER M. // magnitude
	mvSplitsNodeMagSeconds( N, M, M/(ship:availablethrust/ship:mass) ).
}

// node split into magnitude chunks using throttle factor of max
DECLARE FUNCTION mvSplitsNodeMagThrottle{
	DECLARE PARAMETER N. // node number from 1
	DECLARE PARAMETER M. // magnitude
	DECLARE PARAMETER TF. // throttle factor; 1 = 100%
	mvSplitsNodeMagSeconds( N, M, M/(ship:maxthrust*TF/ship:mass) ).
}

// node split into magnitude chunks every T seconds
DECLARE FUNCTION mvSplitsNodeMagSeconds {
	DECLARE PARAMETER N. // node number from 1
	DECLARE PARAMETER M. // magnitude
	DECLARE PARAMETER T. // time between nodes, first is left when it was

	Set sNode to ALLNODES[N-1].
	set MV to M *( mvNodeVec(sNode):NORMALIZED ).
	set nextV to mvNodeVec(sNode).
	UNTIL nextV:MAG <= M {
		set nextV to nextV - MV.
		mvVecNode( sNode, MV ).

		Set newNode to NODE( TIME:SECONDS + ROUND(sNode:ETA) + T, 0, 0, 0 ).
		mvVecNode( newNode, nextV ).
		ADD newNode.
		set sNode to newNode.
	}.
}.

// Add delta to nodes created by mvSplitsNodeMagSeconds
// Use a delta magnitude less than the max node magnitude
// This is for tuning the splitsed nodes.
// It trusts you to give it sensible inputs.
DECLARE FUNCTION mvDeltaSplitsedNodeMagSeconds {
	DECLARE PARAMETER M. // delta magnitude
	DECLARE PARAMETER maxM. // max node magnitude
	DECLARE PARAMETER T. // time between nodes

	SET lNode to ALLNODES[ALLNODES:length-1].
	SET lNodeVec to mvNodeVec( lNode ).
	SET lNodeVecUnit to lNodeVec:Normalized.
	SET targetMag to lNodeVec:MAG + M.
	IF targetMag < 0 {
		SET rM to maxM + targetMag.
		PRINT "UNDERFLOWED by "+rM.
		SET pNode to ALLNODES[ALLNODES:length-2].
		REMOVE lNode.
		mvVecNode( pNode, lNodeVecUnit * rM ).
	} ELSE IF targetMag > maxM {
		SET rM to lNodeVec:MAG + M - maxM.
		PRINT "OVERFLOWED by "+rM.
		mvVecNode( lNode, lNodeVecUnit * maxM ).
		SET nNode to NODE( lNode:ETA + TIME:SECONDS + T, 0,0,0).
		mvVecNode( nNode, rM * lNodeVecUnit ).
		ADD nNode.
	} ELSE {
		mvVecNode( lNode, lNodeVec + M * LNodeVecUnit ).
	}
	WAIT 0.
}.
// todo: the last 'partially full' node can be moved closer to the second last...

DECLARE FUNCTION mvDeltaSplitsedNodeMag{
	DECLARE PARAMETER M. // delta magnitude
	DECLARE PARAMETER maxM. // max node magnitude
	mvDeltaSplitsedNodeMagSeconds( M, maxM, maxM/(ship:availablethrust/ship:mass) ).
}

// node split into magnitude chunks using throttle factor of max
DECLARE FUNCTION mvDeltaSplitsedNodeMagThrottle{
	DECLARE PARAMETER M. // delta magnitude
	DECLARE PARAMETER maxM. // max node magnitude
	DECLARE PARAMETER TF. // throttle factor; 1 = 100%
	mvSplitsNodeMagSeconds( M, maxM, maxM/(ship:maxthrust*TF/ship:mass) ).
}

// Some ion craft can do a full throttle first burn then lesser subsequent burns.
//   1st: pilot knows time = ( eCapcity / ions * ionConsumption) & acc
//   rest know throttle factor & maxTime
//   e.g. splits list( list(300, 1), list( 150, 1), list(180, 0.15) )
//   The last pair is used for any subsequent burns
//   The slew time give MJ a chance to re-point on execute all
DECLARE FUNCTION mvSplitsNodeSecondsThrottleSlew {
	DECLARE PARAMETER N.    // node number from 1
	DECLARE PARAMETER STFs. // list of list( seconds, throttleFactor )
	DECLARE PARAMETER slewSecs is 0. // some buffer time for steering

	Set sNode to ALLNODES[N-1].
	set residualVec to mvNodeVec( sNode ).
	set sUnitVector to residualVec:NORMALIZED.
	set prevBurnTime to 0.
	set refT to sNode:ETA.

	set STFsI to STFs:ITERATOR.
	STFsI:NEXT.
	SET STF to STFsI:VALUE.

	UNTIL residualVec:MAG <= 0.000001 {
		SET maxBurnTime to STF[0].
		SET burnThrottle to STF[1].

		SET actualAcc to ( ship:maxthrust * burnThrottle / ship:mass ).
		SET thisMag to actualAcc * maxBurnTime.
		SET thisDeltaV to sUnitVector * thisMag. // if it's a full burn

		IF residualVec:MAG > thisMag {
			SET residualVec to residualVec - thisDeltaV.
			mvVecNode( sNode, thisDeltaV ).
			SET actualBurnTime to mvNodeVec( sNode ):MAG / actualAcc.
			IF prevBurnTime > 0 {
				// fix sNode time if it's not the first node it's a marker time
				Set actualInterBurn to (prevBurnTime+actualBurnTime)/2.
				Set sNode:ETA to (sNode:ETA - 1 + actualInterBurn + slewSecs).
				WAIT 0.
			}.

			// set residualNode:ETA to disposable value
			Set newNode to NODE( TIME:SECONDS + sNode:ETA + 1, 0, 0, 0 ).
			mvVecNode( newNode, residualVec ).
			ADD newNode.
			WAIT 0.

			set sNode to newNode.
			set prevBurnTime to actualBurnTime.

			STFsI:NEXT.
			IF (NOT STFsI:ATEND) { SET STF to STFsI:VALUE. }.

		} ELSE {
			// find a way to refactor so this is reused code
			SET actualBurnTime to mvNodeVec( sNode ):MAG / actualAcc.
			Set actualInterBurn to (prevBurnTime+actualBurnTime)/2.
			Set sNode:ETA to (sNode:ETA - 1 + actualInterBurn + slewSecs).
			WAIT 0.

			// we are done - tricking condition
			SET residualVec to residualVec - residualVec.
		}.
	}
}.
// todo: try a version that does burn vector (x,y,z) not relative (r,n,p)

// Take a node and split it into PeKick Nodes and the rest.
// Expects a departure node at the right time and place
// It will start kicking before FKWIO & not exceed TotalKickLimitMagnitude
DECLARE FUNCTION mvPeKickNode{
	DECLARE PARAMETER DNODEi. // Departure Node index starting from 1
	DECLARE PARAMETER KM.     // kick magnitude
	DECLARE PARAMETER TKML.   // total kick magnitude limit
	DECLARE PARAMETER FKWIO.  // start kicking within FirstKickWithinOrbits

	SET DNODE to ALLNODES[ DNODEi - 1 ].
	SET TKM to 0.
	SET KV to KM * ( mvNodeVec( DNODE ):NORMALIZED ).
	SET TKVM to 0 * KV.
	SET period to SHIP:ORBIT:PERIOD.

	SET KETA to DNODE:ETA.
	UNTIL KETA <= FKWIO * period { SET KETA to KETA - period. }.

	UNTIL TKM + KM >= TKML {
														//SET po to period.
		SET KETA to KETA + period.
		SET CNode to  NODE( TIME:SECONDS + KETA, KV:X, KV:Y, KV:Z ).
		SET TKM to TKM + KM.
		SET TKVM to TKVM + KV.
		ADD CNode.
		WAIT 0.
		SET period to CNode:ORBIT:PERIOD.
														//PRINT TKM+" < "+TKML+"@"+mvSecondsStr(po)+":"+mvSecondsStr(period).
														//PRINT TKVM:Z.
	}.
	mvVecNode( DNODE, ( mvNodeVec(DNODE) - TKVM ) ).
														//PRINT "mvPeKickNode DONE".
	WAIT 0.
	RETURN TKM.
}.
// todo: a version that computes deltaVPerKick = f( orbit, ship.accNow )
//       maybe make a acc, arc version of kick ( use first kick arc limit to set 
//       kick size and call the other ) + default arc
//       maybe no argument version looks vessel acc mvKickMeLeaving().

// Test Code in LKO 100x100
//     mvClearNodes().
//     ADD NODE( TIME:SECONDS + 60*60*6*5, 0, 0, 1700).
//     wait 0.
//     mvPeKickNode( 100, 850, 3, ALLNODES[0] ).
//     wait 0.
//     mvPrintNodes().
// todo: sanity check FKWIO? incrimental version, kick once more? and unkick? 0 kick to skip an orbit?

// move all nodes in time
DECLARE FUNCTION mvDeltaTAllNodes {
	DECLARE PARAMETER dT. // time to shift
	FOR ANM IN ALLNODES {
		SET ANM:ETA to ANM:ETA + dT.
	}.
}


// Time Utils

DECLARE FUNCTION mvSecondsStr {
	DECLARE PARAMETER s.
	SET sTIME to (TIME + s - TIME).
	RETURN (sTIME:YEAR-1) +"Y"+ (sTIME:DAY-1) +"D " + sTIME:CLOCK.
}.

SET mvDaysPerYear to 426.
SET mvHoursPerDay to 6.
SET mvSecondsPerDay to 60 * 60 * mvHoursPerDay.
SET mvSecondsPerYear to mvSecondsPerDay * mvDaysPerYear.

DECLARE FUNCTION mvAsUTSeconds {
	DECLARE PARAMETER y.
	DECLARE PARAMETER d.
	DECLARE PARAMETER h.
	DECLARE PARAMETER m.
	DECLARE PARAMETER s.
	SET uts to s + m*60 + h*60*60 + (d-1)*mvSecondsPerDay + (y-1)*mvSecondsPerYear.
	RETURN uts.
}.

DECLARE FUNCTION mvUTStr {
	DECLARE PARAMETER s. // ut seconds
	SET y to FLOOR(s/mvSecondsPerYear).
	SET s to s - mvSecondsPerYear*y.

	SET d to FLOOR(s/mvSecondsPerDay).
	SET s to s - mvSecondsPerDay*d.

	SET h to FLOOR(s/(60*60)).
	SET s to s - (60*60)*h.

	SET m to FLOOR(s/60).
	SET s to s - 60*m.

	RETURN "Y"+(y+1) +"D"+ (d+1) +" " + (h)+":"+(m)+":"+ROUND(s).
}.

