using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using UnityEngine;

namespace UnityEditor.TestTools.TestRunner
{
    [Serializable]
    internal class TestFileCleanupVerifier
    {
        const string k_Indent = "    ";

        [SerializeField]
        List<string> m_ExistingFiles;

        [SerializeField]
        bool m_ExistingFilesScanned;

        public Action<object> logAction = Debug.LogWarning;

        private Func<string[]> getAllAssetPathsAction;
        public Func<string[]> GetAllAssetPathsAction
        {
            get
            {
                if (getAllAssetPathsAction != null)
                {
                    return getAllAssetPathsAction;
                }
                return AssetDatabase.GetAllAssetPaths;
            }
            set
            {
                getAllAssetPathsAction = value;
            }
        }

        public void RegisterExistingFiles()
        {
            if (m_ExistingFilesScanned)
            {
                return;
            }

            m_ExistingFiles = GetAllFilesInAssetsDirectory().ToList();
            m_ExistingFilesScanned = true;
        }

        public void VerifyNoNewFilesAdded()
        {
            var currentFiles = GetAllFilesInAssetsDirectory().ToList();

            //Expect that if its the same amount of files, there havent been any changes
            //This is to optimize if there are many files
            if (currentFiles.Count != m_ExistingFiles.Count)
            {
                LogWarningForFilesIfAny(currentFiles.Except(m_ExistingFiles));
            }
        }

        void LogWarningForFilesIfAny(IEnumerable<string> filePaths)
        {
            if (!filePaths.Any())
            {
                return;
            }

            var stringWriter = new StringWriter();
            stringWriter.WriteLine("Files generated by test without cleanup.");
            stringWriter.WriteLine(k_Indent + "Found {0} new files.", filePaths.Count());

            foreach (var filePath in filePaths)
            {
                stringWriter.WriteLine(k_Indent + filePath);
            }

            LogAction(stringWriter.ToString());
        }

        private void LogAction(object obj)
        {
            if (this.logAction != null)
            {
                this.logAction(obj);
            }
            else
            {
                Debug.LogWarning(obj);
            }
        }

        private IEnumerable<string> GetAllFilesInAssetsDirectory()
        {
            return GetAllAssetPathsAction();
        }
    }
}
