﻿using System;
using System.Collections.Generic;
using UnityEngine;

class RcsSounds : PartModule
{
    [KSPField]
    public string rcsSoundFile = "RcsSounds/Sounds/RcsHeavy";
    [KSPField]
    public string rcsShutoffSoundFile = "RcsSounds/Sounds/RcsHeavyShutoff";
    [KSPField]
    public float rcsVolume = 0.5f;
    [KSPField]
    public bool loopRcsSound = true;
    [KSPField]
    public bool internalRcsSoundsOnly = false;
    [KSPField]
    public bool useLightingEffects = true;

    public FXGroup RcsSound = null;
    public FXGroup RcsShutoffSound = null;
    private List<Light> RcsLights = new List<Light>();
    private bool Paused = false;

    private ModuleRCS _rcsModule = null;
    public ModuleRCS rcsModule
    {
        get
        {
            if (this._rcsModule == null)
                this._rcsModule = this.part.FindModuleImplementing<ModuleRCS>();
            return this._rcsModule;
        }
    }

    public override void OnStart(PartModule.StartState state)
    {
        try
        {
            if (state == StartState.Editor || state == StartState.None) return;

            // Works with squad sounds, not with rcsSoundFile.
            if (!GameDatabase.Instance.ExistsAudioClip(rcsSoundFile))
            {
                Debug.LogError("[RcsSounds] Audio file not found: " + rcsSoundFile);
            }

            if (RcsSound != null)
            {
                RcsSound.audio = this.gameObject.AddComponent<AudioSource>();
                RcsSound.audio.dopplerLevel = 0f;
                RcsSound.audio.Stop();
                RcsSound.audio.clip = GameDatabase.Instance.GetAudioClip(rcsSoundFile);
                RcsSound.audio.loop = loopRcsSound;
                // Seek to a random position in the sound file so we don't have 
                // harmonic effects when burning at multiple RCS nozzles.
                RcsSound.audio.time = UnityEngine.Random.Range(0, RcsSound.audio.clip.length);
            }
            else
                Debug.LogError("[RcsSounds] Sound FXGroup not found.");

            if (RcsShutoffSound != null)
            {
                RcsShutoffSound.audio = gameObject.AddComponent<AudioSource>();
                RcsShutoffSound.audio.dopplerLevel = 0f;
                RcsShutoffSound.audio.Stop();
                RcsShutoffSound.audio.clip = GameDatabase.Instance.GetAudioClip(rcsShutoffSoundFile);
                RcsShutoffSound.audio.loop = false;
            }
            else
                Debug.LogError("[RcsSounds] Sound shutoff FXGroup not found.");

            if (useLightingEffects)
                AddLights();

            GameEvents.onGamePause.Add(new EventVoid.OnEvent(OnPause));
            GameEvents.onGameUnpause.Add(new EventVoid.OnEvent(OnUnPause));
        }
        catch (Exception ex)
        {
            Debug.LogError("[RcsSounds] OnStart: " + ex.Message);
        }
    }

    private void AddLights()
    {
        foreach (Transform t in rcsModule.thrusterTransforms)
        {
            // Only one Light is allowed per GameObject, so create a new GameObject each time.
            GameObject rcsLight = new GameObject();
            Light light = rcsLight.AddComponent<Light>();

            light.color = Color.white;
            light.type = LightType.Spot;
            light.intensity = 1f;
            light.range = 2f;
            light.spotAngle = 45f;

            light.transform.parent = t;
            light.transform.position = t.transform.position;
            light.transform.forward = t.transform.up;
            light.enabled = false;
            rcsLight.AddComponent<MeshRenderer>();
            RcsLights.Add(light);
        }
    }

    public void OnDestroy()
    {
        GameEvents.onGamePause.Remove(new EventVoid.OnEvent(OnPause));
        GameEvents.onGameUnpause.Remove(new EventVoid.OnEvent(OnUnPause));
    }

    public void OnPause()
    {
        Paused = true;
        RcsSound.audio.Stop();
        RcsShutoffSound.audio.Stop();
    }

    public void OnUnPause()
    {
        Paused = false;
    }

    private float soundPitch = 1;
    private float soundVolume = 0;
    private bool previouslyActive = false;
    public override void OnUpdate()
    {
        try
        {
            //Debug.Log(String.Format("#Update: Paused: {0}, Audio null: {1}, Sound null: {2}, Shutoff sound null: {3}",
                //Paused, RcsSound == null, RcsSound.audio == null, RcsShutoffSound == null, RcsShutoffSound.audio != null));
            if (!Paused && RcsSound != null && RcsSound.audio != null && RcsShutoffSound != null && RcsShutoffSound.audio != null)
            {
                bool rcsActive = false;
                float strength = 0f;
                //Debug.Log(String.Format("#TestFiring: InternalOnly: {0}, IVA: {1}",
                    //internalRcsSoundsOnly, CameraManager.Instance.currentCameraMode == CameraManager.CameraMode.IVA));
                if (!internalRcsSoundsOnly || CameraManager.Instance.currentCameraMode == CameraManager.CameraMode.IVA)
                {
                    /*/ Check for the resource as the effects still fire slightly without fuel.
                    var resourceList = new List<PartResource>();
                    ResourceFlowMode m;
                    try
                    {
                        m = (ResourceFlowMode)Enum.Parse(typeof(ResourceFlowMode), rcsModule.resourceFlowMode);
                    }
                    catch (Exception)
                    {
                        m = ResourceFlowMode.ALL_VESSEL;
                    }

                    part.GetConnectedResources(PartResourceLibrary.Instance.GetDefinition(rcsModule.resourceName).id,
                        m, resourceList);
                    double totalAmount = 0;
                    foreach (PartResource r in resourceList)
                        totalAmount += r.amount;

                    if (totalAmount >= 0.01) // 0.01 is the smallest amount shown in the resource menu.
                    {*/
                    strength = GetFiringStrength();
                    if (strength > 0.001f)
                        // Don't respond to SAS idling.
                        rcsActive = true;
                    //}
                }

                if (rcsActive)
                {
                    if (rcsModule is ModuleRCSFX)
                    {
                        strength *= 50;
                    }
                    else
                    {
                        for (int i = 0; i < rcsModule.thrusterFX.Count; i++)
                        {
                            if (useLightingEffects)
                            {
                                RcsLights[i].enabled = rcsModule.thrusterFX[i].Active;
                                RcsLights[i].intensity = rcsModule.thrusterFX[i].Power;
                                RcsLights[i].spotAngle = Mathf.Lerp(0, 45, rcsModule.thrusterFX[i].Power);
                            }
                        }
                    }

                    soundVolume = GameSettings.SHIP_VOLUME * rcsVolume * strength;

                    soundPitch = Mathf.Lerp(0.5f, 1f, strength);
                    RcsSound.audio.pitch = soundPitch;
                    RcsSound.audio.volume = soundVolume;
                    if (!RcsSound.audio.isPlaying)
                        RcsSound.audio.Play();
                    previouslyActive = true;
                }
                else
                {
                    RcsSound.audio.Stop();
                    if (useLightingEffects)
                    {
                        for (int i = 0; i < rcsModule.thrusterFX.Count; i++)
                            RcsLights[i].enabled = false;
                    }
                    if (previouslyActive)
                    {
                        if (!internalRcsSoundsOnly ||
                            CameraManager.Instance.currentCameraMode == CameraManager.CameraMode.IVA)
                        {
                            RcsShutoffSound.audio.volume = soundVolume / 2;
                            RcsShutoffSound.audio.Play();
                        }
                        previouslyActive = false;
                    }
                }
            }
        }
        catch (Exception ex)
        {
            Debug.LogError("[RcsSounds] Error OnUpdate: " + ex.Message);
        }
    }

    private float GetFiringStrength()
    {
        if (rcsModule is ModuleRCSFX)
        {
            ModuleRCSFX rcsfxModule = rcsModule as ModuleRCSFX;
            float maxVolume = 0;

            EffectBehaviour[] effectBehaviours = part.GetComponents<EffectBehaviour>();
            for (int i = 0; i < effectBehaviours.Length; i++)
            {
                if (effectBehaviours[i].effectName == rcsfxModule.runningEffectName)
                {
                    AudioSource[] audioSources = effectBehaviours[i].GetComponentsInChildren<AudioSource>();
                    if (audioSources.Length > 0)
                    {
                        for (int j = 0; j < audioSources.Length; j++)
                        {
                            // GetComponentsInChildren will return this module's AudioSources too.
                            if (audioSources[j].clip != null &&
                                audioSources[j].clip.name != rcsSoundFile && audioSources[j].clip.name != rcsShutoffSoundFile)
                            {
                                if (audioSources[j].isPlaying)
                                {
                                    maxVolume = Mathf.Max(maxVolume, audioSources[j].volume);
                                }
                            }
                        }
                    }
                }
            }
            return maxVolume;
        }
        else
        {
            float rcsHighestPower = 0;
            for (int i = 0; i < rcsModule.thrusterFX.Count; i++)
            {
                rcsHighestPower = Mathf.Max(rcsHighestPower, rcsModule.thrusterFX[i].Power);
            }
            return rcsHighestPower;
        }
    }
}
