﻿// Kitten Engineer Redux v0.4.0
// BarneyTheGod - StarMap v0.3.4
// This mod will serve an entirely different purpose when the KSA devs eventually add the VAB, but for now it provides in-flight engineering tools.

using Brutal.ImGuiApi;
using Brutal.Numerics;
using HarmonyLib;
using KSA;
using ModMenu;
using StarMap.API;
using System;
using System.Reflection;

namespace KittenEngineerRedux
{
    [StarMapMod]
    public class Class1
    {
        private const string HarmonyId = "com.kittenengineerredux.starmap";
        private readonly Harmony _harmony = new Harmony(HarmonyId);
        private bool _showWindow = true;
        private int _currentTab = 0;
        private double _engineThrust = 100000.0;
        private double _engineIsp = 350.0;
        private const double StandardGravity = 9.80665;
        private int _circularizeLocation = 0;
        private bool _circularizeBurnActive = false;
        private bool _timeWarpToCircularize = false;
        private double _targetAltitude = 0.0;
        private bool _burnEngaged = false;
        private bool _warpInitiated = false;

        [StarMapAllModsLoaded]
        public void OnFullyLoaded()
        {
            try { _harmony.PatchAll(Assembly.GetExecutingAssembly()); }
            catch { }
        }

        [StarMapUnload]
        public void Unload() => _harmony.UnpatchAll(HarmonyId);

        // MODMENU INTEGRATION
        [ModMenuEntry("Kitten Engineer Redux")]
        public void ModMenuEntry()
        {
            Vehicle? vehicle = Program.ControlledVehicle;
            if (vehicle != null)
            {
                Astronomical parent = vehicle.Orbit.Parent;
                double bodyRadius = parent.MeanRadius;

                if (ImGui.BeginMenu("KER Panel"))
                {
                    if (ImGui.BeginTabBar("MainTabs", ImGuiTabBarFlags.None))
                    {
                        if (ImGui.BeginTabItem("Telemetry"))
                        {
                            DrawTelemetry(vehicle, parent);
                            ImGui.EndTabItem();
                        }

                        if (ImGui.BeginTabItem("Suicide Burn"))
                        {
                            DrawSuicideBurn(vehicle, parent, bodyRadius);
                            ImGui.EndTabItem();
                        }

                        if (ImGui.BeginTabItem("Flight Computer"))
                        {
                            DrawFlightComputerUi(vehicle, parent, bodyRadius);
                            ImGui.EndTabItem();
                        }

                        if (ImGui.BeginTabItem("Settings"))
                        {
                            DrawSettings();
                            ImGui.EndTabItem();
                        }

                        ImGui.EndTabBar();
                    }
                    ImGui.EndMenu();
                }
            }
            else
            {
                if (ImGui.BeginMenu("Kitten Engineer Redux"))
                {
                    ImGui.TextColored(new float4(1.0f, 0.5f, 0.5f, 1.0f), "No vehicle controlled");
                    ImGui.EndMenu();
                }
            }
        }

        [StarMapAfterGui]
        public void OnAfterUi(double dt)
        {
            DrawUi();
        }

        private void DrawUi()
        {
            if (!_showWindow) return;

            ImGui.PushStyleVar(ImGuiStyleVar.WindowRounding, 8.0f);
            ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, 6.0f);
            ImGui.PushStyleVar(ImGuiStyleVar.TabRounding, 6.0f);

            ImGui.Begin("Kitten Engineer Redux", ref _showWindow, ImGuiWindowFlags.None);

            Vehicle? vehicle = Program.ControlledVehicle;
            if (vehicle != null)
            {
                Astronomical parent = vehicle.Orbit.Parent;
                double bodyRadius = parent.MeanRadius;

                if (ImGui.BeginTabBar("MainTabs", ImGuiTabBarFlags.None))
                {
                    if (ImGui.BeginTabItem("Telemetry"))
                    {
                        DrawTelemetry(vehicle, parent);
                        ImGui.EndTabItem();
                    }

                    if (ImGui.BeginTabItem("Suicide Burn"))
                    {
                        DrawSuicideBurn(vehicle, parent, bodyRadius);
                        ImGui.EndTabItem();
                    }

                    if (ImGui.BeginTabItem("Flight Computer"))
                    {
                        DrawFlightComputerUi(vehicle, parent, bodyRadius);
                        ImGui.EndTabItem();
                    }

                    if (ImGui.BeginTabItem("Settings"))
                    {
                        DrawSettings();
                        ImGui.EndTabItem();
                    }

                    ImGui.EndTabBar();
                }
            }
            else
            {
                ImGui.TextColored(new float4(1.0f, 0.5f, 0.5f, 1.0f), "No vehicle controlled");
            }

            ImGui.End();
            ImGui.PopStyleVar(3);
        }

        private void DrawTelemetry(Vehicle vehicle, Astronomical parent)
        {
            NavBallData navBallData = vehicle.NavBallData;
            double bodyRadius = parent.MeanRadius;

            ImGui.Spacing();
            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Vessel Status");
            ImGui.Separator();
            ImGui.Text($"Altitude : {navBallData.Altitude:N0} m");
            ImGui.Text($"Speed    : {navBallData.Speed:F1} m/s");

            double currentThrottle = GetEngineThrottle(vehicle);
            double currentThrust = currentThrottle * _engineThrust;
            double accelMps = currentThrust / vehicle.TotalMass;
            double gForce = accelMps / StandardGravity;

            float4 gColor = new float4(0, 1, 0, 1);
            if (gForce > 4.0) gColor = new float4(1, 1, 0, 1);
            if (gForce > 9.0) gColor = new float4(1, 0, 0, 1);

            ImGui.Text("G-Force  : ");
            ImGui.SameLine();
            ImGui.TextColored(gColor, $"{gForce:F2} G");

            ImGui.Spacing();
            ImGui.Spacing();
            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Orbital Parameters");
            ImGui.Separator();

            double apAltitude = vehicle.Apoapsis - bodyRadius;
            double peAltitude = vehicle.Periapsis - bodyRadius;

            ImGui.Text($"Apoapsis : {apAltitude:N0} m");
            ImGui.Text($"Periapsis: {peAltitude:N0} m");

            SimTime currentTime = Universe.GetElapsedSimTime();
            double timeToAp = (vehicle.NextApoapsisTime - currentTime).Seconds();
            double timeToPe = (vehicle.NextPeriapsisTime - currentTime).Seconds();

            ImGui.Text($"T- Ap    : {FormatTime(timeToAp)}");
            ImGui.Text($"T- Pe    : {FormatTime(timeToPe)}");

            ImGui.Spacing();
            ImGui.Spacing();
            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Vessel Info");
            ImGui.Separator();
            ImGui.Text($"Mass     : {vehicle.TotalMass:N0} kg");
            ImGui.Text($"TWR      : {vehicle.NavBallData.ThrustWeightRatio:F2}");
        }

        private void DrawSuicideBurn(Vehicle vehicle, Astronomical parent, double bodyRadius)
        {
            ImGui.Spacing();

            double3 positionCci = vehicle.GetPositionCci();
            double altitude = positionCci.Length() - bodyRadius;
            double3 posNorm = positionCci.Normalized();
            double verticalSpeed = -double3.Dot(vehicle.GetVelocityCci(), posNorm);
            if (verticalSpeed <= 0.0) verticalSpeed = 0.0;

            double twr = vehicle.NavBallData.ThrustWeightRatio;
            double r_vessel = bodyRadius + altitude;
            double localGravity = 6.6743E-11 * parent.Mass / (r_vessel * r_vessel);

            var (timeToStop, distanceToStop, netDecel) = CalculateStoppingParameters(verticalSpeed, twr, localGravity);

            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Landing Analysis");
            ImGui.Separator();
            ImGui.Text($"Altitude        : {altitude:N0} m");
            ImGui.Text($"Vertical Speed  : {verticalSpeed:F1} m/s");
            ImGui.Text($"Stop Distance   : {distanceToStop:F1} m");
            ImGui.Text($"Stop Time       : {timeToStop:F1} s");

            ImGui.Spacing();

            if (altitude > 5 && verticalSpeed > 1.0)
            {
                if (distanceToStop >= altitude)
                {
                    ImGui.Spacing();
                    ImGui.PushStyleColor(ImGuiCol.Text, 0xFF0000FF);
                    ImGui.Text("!!! BURN NOW !!!");
                    ImGui.PopStyleColor();
                }
                else
                {
                    double margin = altitude - distanceToStop;
                    float4 marginColor = margin > 500 ? new float4(0, 1, 0, 1) : new float4(1, 1, 0, 1);
                    ImGui.Text("Safety Margin   : ");
                    ImGui.SameLine();
                    ImGui.TextColored(marginColor, $"{margin:F1} m");
                }
            }
            else
            {
                ImGui.TextColored(new float4(0.7f, 0.7f, 0.7f, 1.0f), "Not descending or altitude too low");
            }
        }

        private void DrawFlightComputerUi(Vehicle vehicle, Astronomical parent, double bodyRadius)
        {
            ImGui.Spacing();
            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Circularize Orbit Autopilot");
            ImGui.Separator();

            string[] locations = new string[] { "At Apoapsis", "At Periapsis" };
            if (ImGui.BeginCombo("Burn Location", locations[_circularizeLocation]))
            {
                for (int i = 0; i < locations.Length; i++)
                {
                    bool isSelected = (_circularizeLocation == i);
                    if (ImGui.Selectable(locations[i], isSelected))
                        _circularizeLocation = i;
                    if (isSelected) ImGui.SetItemDefaultFocus();
                }
                ImGui.EndCombo();
            }

            ImGui.Checkbox("Auto-Warp to Burn", ref _timeWarpToCircularize);
            ImGui.Spacing();

            if (_circularizeBurnActive)
            {
                ImGui.PushStyleColor(ImGuiCol.Button, 0xFF0000FF);
                if (ImGui.Button("Abort Circularization", new float2(200, 30)))
                {
                    vehicle.SetEnum(VehicleEngine.MainShutdown);
                    _circularizeBurnActive = false;
                    _burnEngaged = false;
                    _warpInitiated = false;
                    vehicle.FlightComputer.AttitudeMode = FlightComputerAttitudeMode.Manual;
                    if (Universe.AutoWarpActive) Universe.AutoWarpStop(true);
                }
                ImGui.PopStyleColor();
            }
            else
            {
                ImGui.PushStyleColor(ImGuiCol.Button, 0xFF00AA00);
                if (ImGui.Button("Initiate Circularization", new float2(200, 30)))
                {
                    _circularizeBurnActive = true;
                    _burnEngaged = false;
                    _warpInitiated = false;

                    double currentAp = vehicle.Apoapsis;
                    double currentPe = vehicle.Periapsis;
                    bool isAp = _circularizeLocation == 0;
                    double targetR = isAp ? currentAp : currentPe;
                    _targetAltitude = targetR - bodyRadius;

                    FlightComputer fc = vehicle.FlightComputer;
                    fc.AttitudeMode = FlightComputerAttitudeMode.Auto;
                    fc.AttitudeTrackTarget = isAp ? FlightComputerAttitudeTrackTarget.Prograde : FlightComputerAttitudeTrackTarget.Retrograde;
                    fc.AttitudeFrame = VehicleReferenceFrame.EclBody;

                    if (_timeWarpToCircularize)
                    {
                        SimTime t = isAp ? vehicle.NextApoapsisTime : vehicle.NextPeriapsisTime;
                        Universe.AutoWarpTo(t);
                        _warpInitiated = true;
                    }
                }
                ImGui.PopStyleColor();
            }

            ImGui.Spacing();
            ImGui.Separator();

            if (_circularizeBurnActive || vehicle.Apoapsis > vehicle.Orbit.Parent.MeanRadius)
            {
                double curAp = vehicle.Apoapsis;
                double curPe = vehicle.Periapsis;
                bool atAp = _circularizeLocation == 0;
                double circRadius = atAp ? curAp : curPe;
                double sma = (curAp + curPe) / 2.0;
                double mu = 6.6743E-11 * parent.Mass;

                double vCirc = Math.Sqrt(mu / circRadius);
                double vCurrent = Math.Sqrt(mu * (2.0 / circRadius - 1.0 / sma));
                double dV = Math.Abs(vCirc - vCurrent);

                ImGui.Text($"Required dV: {dV:F1} m/s");

                if (_circularizeBurnActive)
                {
                    if (_burnEngaged)
                        ImGui.TextColored(new float4(0, 1, 0, 1), "STATUS: BURNING");
                    else if (_warpInitiated)
                        ImGui.TextColored(new float4(1, 1, 0, 1), "STATUS: WARPING");
                    else
                        ImGui.TextColored(new float4(0.7f, 0.7f, 1.0f, 1.0f), "STATUS: Aligning...");
                }
            }

            if (_circularizeBurnActive)
            {
                SimTime now = Universe.GetElapsedSimTime();
                bool atAp = _circularizeLocation == 0;
                double timeToTg = atAp ? (vehicle.NextApoapsisTime - now).Seconds() : (vehicle.NextPeriapsisTime - now).Seconds();

                if (!_burnEngaged && timeToTg < 5.0)
                {
                    _burnEngaged = true;
                    if (Universe.AutoWarpActive) Universe.AutoWarpStop(true);
                }

                if (_burnEngaged)
                {
                    double curAp = vehicle.Apoapsis;
                    double curPe = vehicle.Periapsis;
                    double oppAlt = atAp ? (curPe - bodyRadius) : (curAp - bodyRadius);
                    double targetAltR = _targetAltitude;

                    if (Math.Abs(targetAltR - oppAlt) < 1000)
                    {
                        SetEngineThrottle(vehicle, 0.0);
                        vehicle.SetEnum(VehicleEngine.MainShutdown);
                        _circularizeBurnActive = false;
                        _burnEngaged = false;
                        vehicle.FlightComputer.AttitudeMode = FlightComputerAttitudeMode.Manual;
                    }
                    else
                    {
                        SetEngineThrottle(vehicle, 1.0);
                        vehicle.SetEnum(VehicleEngine.MainIgnite);
                    }
                }
            }
        }

        private void DrawSettings()
        {
            ImGui.Spacing();
            ImGui.TextColored(new float4(0.4f, 1.0f, 0.4f, 1.0f), "Engine Parameters");
            ImGui.Separator();

            ImGui.Text("These settings affect G-force calculations");
            ImGui.Spacing();

            ImGui.InputDouble("Max Engine Thrust (N)", ref _engineThrust);
            ImGui.InputDouble("Engine ISP (s)", ref _engineIsp);

            ImGui.Spacing();
            ImGui.Separator();
            ImGui.TextColored(new float4(0.7f, 0.7f, 0.7f, 1.0f), "Kitten Engineer Redux v0.4.0");
        }

        private string FormatTime(double seconds)
        {
            if (double.IsNaN(seconds) || double.IsInfinity(seconds)) return "N/A";
            if (seconds < 0) return "N/A";
            if (seconds > 3600) return $"{seconds / 3600.0:F1} h";
            if (seconds > 60) return $"{seconds / 60.0:F1} m";
            return $"{seconds:F1} s";
        }

        private (double timeToStop, double distanceToStop, double netDeceleration)
          CalculateStoppingParameters(double v, double twr, double g)
        {
            if (twr <= 0 || v <= 0) return (0, 0, 0);

            double a_thrust = twr * g;
            double a = a_thrust - g;
            if (a <= 0.0) return (double.MaxValue, double.MaxValue, a);
            double tStop = v / a;
            double dStop = (v * v) / (2.0 * a);
            return (tStop, dStop, a);
        }

        private void SetEngineThrottle(Vehicle vehicle, double throttleValue)
        {
            FieldInfo? manualControlInputsField = typeof(Vehicle).GetField("_manualControlInputs", BindingFlags.NonPublic | BindingFlags.Instance);
            if (manualControlInputsField == null) return;

            object? controlInputs = manualControlInputsField.GetValue(vehicle);
            if (controlInputs == null) return;

            PropertyInfo? engineThrottleProp = controlInputs.GetType().GetProperty("EngineThrottle");
            FieldInfo? engineThrottleField = controlInputs.GetType().GetField("EngineThrottle");

            float clampedValue = (float)Math.Clamp(throttleValue, 0.0, 1.0);

            if (engineThrottleProp != null)
            {
                engineThrottleProp.SetValue(controlInputs, clampedValue);
            }
            else if (engineThrottleField != null)
            {
                engineThrottleField.SetValue(controlInputs, clampedValue);
            }
        }

        private double GetEngineThrottle(Vehicle vehicle)
        {
            FieldInfo? manualControlInputsField = typeof(Vehicle).GetField("_manualControlInputs", BindingFlags.NonPublic | BindingFlags.Instance);
            if (manualControlInputsField == null) return 0.0;

            object? controlInputs = manualControlInputsField.GetValue(vehicle);
            if (controlInputs == null) return 0.0;

            PropertyInfo? engineThrottleProp = controlInputs.GetType().GetProperty("EngineThrottle");
            FieldInfo? engineThrottleField = controlInputs.GetType().GetField("EngineThrottle");

            if (engineThrottleProp != null)
            {
                return (float)(engineThrottleProp.GetValue(controlInputs) ?? 0.0f);
            }
            else if (engineThrottleField != null)
            {
                return (float)(engineThrottleField.GetValue(controlInputs) ?? 0.0f);
            }
            return 0.0;
        }
    }
}