#version 450

#include "../../Core/Shaders/Common/Global.glsl"
#include "../../Core/Shaders/Common/Gauges.glsl"

const int LAT_STEPS = 3;
const int AZI_STEPS = 12;
const int HEMISPHERES = 2;
const int SIDE_TICKS = 2;

const float TICK_LENGTH_RAD = radians(1);
const float OFFSET_RAD = radians(5.0);
const float STANDARD_EXTENDS = 128.0;

vec3 rotateAroundAxis(vec3 v, vec3 k, float theta) {
    return v * cos(theta)
    + cross(k, v) * sin(theta)
    + k * dot(k, v) * (1.0 - cos(theta));
}

int getDegreesYaw(vec3 center, vec3 axisCross, float pushOutRad) {
    vec3 dir = normalize(center + axisCross * pushOutRad);
    if (dot(center, dir) <= 0.0)
        return 0;
    float rawAngle = degrees(acos(dot(dir, vec3(0.0, -1.0, 0.0)))) - 90.0;
    return int(mod(round(rawAngle), 360.0));
}

int getDegreesPitch(vec3 center, vec3 axisCross, float pushOutRad) {
    vec3 dir = normalize(center + axisCross * pushOutRad);
    if (dot(center, dir) <= 0.0) 
        return 0;
    float rawDeg = degrees( atan(dir.z, dir.x) );
    float wrapped = mod(rawDeg - 180, 360.0);
    return int(floor(wrapped + 0.5));
}

float drawIntOnSphere(
    sampler2D atlas,
    int       value,
    vec3      normal,
    vec3      center,
    vec3      axisCrossC,
    float     pushOutRad,
    float     textScale,   // in radians per glyph unit
    inout float outAlpha
) 
{
    vec3 labelCenter = normalize(center + pushOutRad * axisCrossC);
    if (dot(normal, labelCenter) <= 0.0) return outAlpha;
    vec3 T = normalize(cross(vec3(0,1,0), labelCenter));
    vec3 B = normalize(cross(labelCenter, T));

    // unpack digits
    int  digits[4];
    int  n = 0;
    int  tmp = value;
    if (tmp == 0) {
        digits[n++] = 0;
    } else {
        while (tmp > 0 && n < 4) {
            digits[n++] = tmp % 10;
            tmp /= 10;
        }
    }
    // reverse
    for (int i = 0; i < n/2; ++i) {
        int t = digits[i];
        digits[i] = digits[n-1-i];
        digits[n-1-i] = t;
    }

    // total angular width & height
    float totalAdv = 0.0;
    for (int i = 0; i < n; ++i) {
        totalAdv += GLYPHS[digits[i]].adv;
        if (i < n-1)      // only add spacing between glyphs
        totalAdv += SPACING;
    }
    float widthRad  = totalAdv * textScale;
    float heightRad = 1.0 * textScale;

    // compute angular offsets
    float px0 = dot(normal, T);
    float py0 = dot(normal, B);

    float px  = py0;
    float py  = -px0;
    
    float pz = dot(normal, labelCenter);
    float angX = atan(px, pz);
    float angY = atan(py, pz);

    // place & rasterize each glyph
    float firstAdv  = GLYPHS[digits[0]].adv * textScale;
    float cursorAng = -0.5 * widthRad + 0.5 * firstAdv;
    for (int i = 0; i < n; ++i) {
        Glyph g       = GLYPHS[digits[i]];
        float advRad  = g.adv * textScale;

        float u = (angX - cursorAng) / advRad + 0.5;
        float v = 1.0 - ((angY + 0.5 * heightRad) / heightRad);

        if (u >= 0.0 && u <= 1.0 && v >= 0.0 && v <= 1.0) {
            vec2 atlasUV = mix(g.uv.xy, g.uv.zw, vec2(u, v));
            float navballPx = pushConsts.extends;
            float glyphScale = max(advRad,heightRad) * navballPx / global.camera.sdfResolution;
            float a = sampleMTSDF_AA_px(atlas, atlasUV, glyphScale);
            outAlpha  = max(outAlpha, a);
        }

        // new, correct: half this glyph + spacing + half next glyph
        if (i < n-1) {
            float nextAdv = GLYPHS[digits[i+1]].adv * textScale;
            cursorAng += 0.5 * advRad
                   + SPACING * textScale
                    + 0.5 * nextAdv;
        } else {
            // last glyph: move just half out so nothing breaks
            cursorAng += 0.5 * advRad;
        }
    }

    return outAlpha;
}

const float SIN_LAT[3] = float[3](
    0.2588190451,   // sin( 15°)
    0.7071067812,   // sin( 45°)
    0.9659258263    // sin( 75°)
);
const float COS_LAT[3] = float[3](
    0.9659258263,   // cos( 15°)
    0.7071067812,   // cos( 45°)
    0.2588190451    // cos( 75°)
);

const float COS_AZI[12] = float[12](
    0.9659258263,  0.7071067812,  0.2588190451, -0.2588190451,
    -0.7071067812, -0.9659258263, -0.9659258263, -0.7071067812,
    -0.2588190451,  0.2588190451,  0.7071067812,  0.9659258263
);
const float SIN_AZI[12] = float[12](
    0.2588190451,  0.7071067812,  0.9659258263,  0.9659258263,
    0.7071067812,  0.2588190451, -0.2588190451, -0.7071067812,
    -0.9659258263, -0.9659258263, -0.7071067812, -0.2588190451
);

layout(location = 0) out vec4 outColor;

const float CUTOFF = 0.50; // how much of the sphere to cut
const float CUTOFF_KEEP = 1.0 - 0.50;
const float POLE_RING = cos(radians(5.0));

// don’t draw anything beyond this latitude
const float POLE_CUTOFF_RAD = radians(80.0);
const float POLE_Y_CUTOFF   = sin(POLE_CUTOFF_RAD);

// wrapper that skips anything whose centerDir lies past cutoff
float drawGreatCircleSegmentClamped(vec3 dir, vec3 normal, vec3 centerDir, float maxAngleRad, float thicknessPx, float cutoff)
{
    if (abs(centerDir.y) > cutoff)
        return 0.0;
    return drawGreatCircleSegment(dir, normal, centerDir, maxAngleRad, thicknessPx);
}

float drawTFillPS(vec2 pix, float lenPx, float thkPx)
{
    float halfLen = 0.5 * lenPx;

    // horizontal bar at y=0, from x=±halfLen (unchanged)
    float hor = aaBand1D(abs(pix.y), thkPx)
    * aaCap1D(abs(pix.x), halfLen);

    // vertical stem going *up* from y=0 to y=+halfLen:
    //   step(0.0,pix.y)  == 1 when pix.y >=  0
    //   step(halfLen,pix.y) == 1 when pix.y >= halfLen
    // so step(0) - step(halfLen) is 1 only in [0…halfLen)
    float inY = clamp( step(0.0, pix.y) - step(halfLen, pix.y), 0.0, 1.0 );
    float ver = aaBand1D(abs(pix.x), thkPx) * inY;

    return max(hor, ver);
}

// builds a slightly fatter T and subtracts the inner one
float drawTBorderPS(vec2 pix, float lenPx, float thkPx, float borderPx)
{
    float outerLen = lenPx + 2.0*borderPx;
    float outerThk = thkPx + 2.0*borderPx;

    float big   = drawTFillPS(pix, outerLen, outerThk);
    float small = drawTFillPS(pix, lenPx,    thkPx);
    return clamp(big - small, 0.0, 1.0);
}

void main()
{
    // given input UV in the full screen-space
    vec2 uv = inUv * 2.0 - 1.0;
    
    float distSq = dot(uv, uv);
    float keepSq = 1.0 - CUTOFF_KEEP * CUTOFF_KEEP;
    if (distSq > keepSq)
        discard;

    float thicknessPx = pushConsts.extends / STANDARD_EXTENDS;
    float halfThickness = thicknessPx * 0.5;

    float x = uv.x;
    float y = uv.y;
    float z = sqrt(1.0 - distSq);
    vec3 preRot = vec3(x, y, z);
    mat4 defaultRotation = mat4(1.0);
    vec3 rotated = (global.vessel.localRotation * vec4(preRot, 0.0)).xyz;
    vec3 normal  = normalize(rotated);

    // how we find the color
    vec3 finalColor;

    // Color the rest of the sphere
    vec3 skyColor    = vec3(0.349, 0.404, 0.439);
    vec3 groundColor = vec3(0.09, 0.102, 0.11);
    vec3 black = vec3(0);
    vec3 white = vec3(1);
    vec3 faceColor = normal.z < 0.0 ? skyColor : groundColor;
    finalColor = faceColor;

    float navballPx   = pushConsts.extends;
    vec3 lineCol = normal.z < 0.0 ? black : white;

    // horizontal lines going around the sphere
    float horizontalLines = 0.0;
    for (int i = 1; i < 6; ++i) {
        float a = float(i) * 6.28318530718 / 12.0;
        vec3 n = vec3( sin(a), 0.0, cos(a) ); // now sweeping around Z-axis
        horizontalLines = max(horizontalLines, drawGreatCircle(normal, n, thicknessPx));
    }

    // azimuth lines going around the sphere
    float azimuthRings = 0.0;
    for (int i = 1; i <= 3; ++i) {
        float azimuthDeg = 30.0 * float(i);
        azimuthRings = max(azimuthRings, drawAzimuthRing(normal,  azimuthDeg, thicknessPx));
        azimuthRings = max(azimuthRings, drawAzimuthRing(normal, -azimuthDeg, thicknessPx));
    }

    // combine grid & pick line color per hemisphere
    float lines = max(horizontalLines, azimuthRings);
    finalColor = mix(finalColor, lineCol, lines);

    // draw the horizon line
    float horizon = drawGreatCircle(normal, AXIS_Z, thicknessPx * 0.5);
    finalColor = mix(finalColor, groundColor, horizon);

    // central pitch marker helper lines
    float tickMarks = drawLineMarks(normal, 2, 2.0, thicknessPx);
    tickMarks = max(tickMarks, drawLineMarks(normal, 10, 3.0, thicknessPx));
    finalColor = mix(finalColor, lineCol, tickMarks);
    
    // mid grid helper markers
    float crossTicks = 0.0;
    for (int lat = 0; lat < LAT_STEPS; ++lat) {
        float y0 = SIN_LAT[lat];
        float r0 = COS_LAT[lat];
        for (int a = 0; a < AZI_STEPS; ++a) {
            float x0 = r0 * COS_AZI[a];
            float z0 = r0 * SIN_AZI[a];
            float pushOutH  = OFFSET_RAD * float(SIDE_TICKS+1);
            float pushOutV  = OFFSET_RAD * r0 * float(SIDE_TICKS+1);
            for (int h = 0; h < HEMISPHERES; ++h) {
                float sY = h == 0 ? 1.0 : -1.0;
                vec3 center = vec3(x0, y0 * sY, z0);
                vec3 right   = normalize(cross(AXIS_Y, center));
                vec3 forward = normalize(cross(center, right));
                vec3 axisCrossC = cross(right, center);
                vec3 axisCross = normalize(cross(forward, center));

                // pre‐hoisted
                float scaledOffset = OFFSET_RAD * r0;

                // center cross
                vec3 nR = cross(center, right);
                vec3 nF = cross(center, forward);
                crossTicks = max(crossTicks,drawGreatCircleSegment(normal, nR, center, TICK_LENGTH_RAD, halfThickness));
                float boxSize = radians(5);
                
                if (lat != 2) {
                    crossTicks = max(crossTicks,
                    drawGreatCircleSegment(normal, nF, center, TICK_LENGTH_RAD, halfThickness));

                    // vertical side‐ticks (–)
                    for (int s = 1; s <= SIDE_TICKS; ++s) {
                        float o = scaledOffset * float(s);
                        vec3 axisCrossC = cross(forward, center);
                        vec3 offL = normalize(center + o * axisCrossC);
                        vec3 offR = normalize(center - o * axisCrossC);
                        vec3 fL = normalize(cross(offL, cross(AXIS_Y, offL)));
                        vec3 fR = normalize(cross(offR, cross(AXIS_Y, offR)));
                        vec3 nL = cross(offL, fL);
                        vec3 nR2= cross(offR, fR);
                        crossTicks = max(crossTicks,
                        drawGreatCircleSegment(normal, nL, offL, TICK_LENGTH_RAD, halfThickness));
                        crossTicks = max(crossTicks,
                        drawGreatCircleSegment(normal, nR2, offR, TICK_LENGTH_RAD, halfThickness));
                        
                    }
                }

                // horizontal side‐ticks (|) — ringRadius is always 1 so
                for (int s = 1; s <= SIDE_TICKS; ++s) {
                    float o = OFFSET_RAD * float(s);
                    vec3 offF = normalize(center + o * axisCrossC);
                    vec3 offB = normalize(center - o * axisCrossC);
                    vec3 rF = normalize(cross(AXIS_Y, offF));
                    vec3 rB = normalize(cross(AXIS_Y, offB));
                    vec3 nF2= cross(offF, rF);
                    vec3 nB2= cross(offB, rB);
                    crossTicks = max(crossTicks,drawGreatCircleSegmentClamped(normal, nF2, offF, TICK_LENGTH_RAD, halfThickness, POLE_Y_CUTOFF));
                    crossTicks = max(crossTicks,drawGreatCircleSegmentClamped(normal, nB2, offB, TICK_LENGTH_RAD, halfThickness, POLE_Y_CUTOFF));    
                }
                const float fontScale = 0.07;

                // label vertical
                float alphaV = 0.0;
                float offV = scaledOffset * float(SIDE_TICKS + 1);
                int value = abs(getDegreesPitch(center, axisCross, offV)) / 10;
                if (value != 36) 
                {
                    vec3 dir = normalize(center + axisCross * offV);
                    if (abs(dir.y) < sin(radians(60)))
                    {
                        vec3 faceColor2 = faceColor;
                        vec3 lineColor2 = lineCol;
                        if (value == 18 || value == 0)
                        {
                            faceColor2 = skyColor;
                            lineColor2 = black;
                            float maskV = labelRoundedBoxMask(normal, center, axisCross, offV, boxSize, radians(2.5), pushConsts.extends);
                            finalColor = mix(finalColor, faceColor2, maskV);
                        }
                        else
                        {
                            float maskV = labelBoxMask(normal, center, axisCross, offV, boxSize, pushConsts.extends);
                            finalColor = mix(finalColor, faceColor2, maskV);
                        }
                        
                        drawIntOnSphere(atlas, value, normal, center, axisCross, offV, fontScale, alphaV);
                        finalColor = mix(finalColor, lineColor2, alphaV);
                    }
                }

                // label the slices
                int yaw = getDegreesYaw(center, axisCrossC, pushOutH) / 10;
                if (yaw != 0 && yaw != 36 && yaw != 27)
                {
                    {
                        float mask = labelBoxMask(
                            normal,
                            center,
                            axisCrossC,
                            OFFSET_RAD * float(SIDE_TICKS + 1), // push out one more step
                            boxSize,
                            pushConsts.extends
                        );
                        finalColor = mix(finalColor, faceColor, mask);

                        float alphaH = 0.0;
                        drawIntOnSphere(atlas, yaw, normal, center, axisCrossC, pushOutH, fontScale, alphaH);
                        finalColor = mix(finalColor, lineCol, alphaH);
                    }
                }
            }
        }
    }
    finalColor = mix(finalColor, lineCol, crossTicks);

    // draw the pole ring
    finalColor = mix(finalColor, lineCol, drawAzimuthRing(normal, 85, thicknessPx));
    finalColor = mix(finalColor, lineCol, drawAzimuthRing(normal, -85, thicknessPx));

    // draw a central pitch line around the sphere
    float invR     = 1.0 / pushConsts.extends;
    float center = aaLinePix(abs(dot(normal, AXIS_Y)), thicknessPx * 2, invR);
    finalColor     = mix(finalColor, lineCol, center);

    //finalColor = mix(finalColor, lineCol, drawAzimuthRing(normal, 1, thicknessPx * 0.5));
    //finalColor = mix(finalColor, lineCol, drawAzimuthRing(normal, -1, thicknessPx * 0.5));

    // 1) get the real screen‐pixel of this fragment
    vec2 resolution = vec2(global.camera.screenWidth, global.camera.screenHeight);
    vec2 pixPos      = uv * pushConsts.extends; // Pixel coordinate in UV space

    const float CROSS_LENGTH = 0.3;  // crossbar length
    const float CROSS_THICKNESS = 0.02; // arm thickness
    const float BORDER_PX = 1.0;
    
    float crossPx  = pushConsts.extends * CROSS_LENGTH;
    float thickPx  = pushConsts.extends * CROSS_THICKNESS;
    
    float fillMask   = drawTFillPS(  pixPos, crossPx, thickPx);
    finalColor = mix(finalColor, vec3(1.0), fillMask);
    
    float borderMask = drawTBorderPS(pixPos, crossPx, thickPx, BORDER_PX);
    finalColor = mix(finalColor, vec3(0.0), borderMask);

    // Color the output
    outColor = vec4(finalColor, 1.0);
}
