# KSP Plugin Loader

This is a KSP mod that allows `KSPAssemblyDependency` attributes to be used to
depend on other KSP mods that do not have a `KSPAssembly` attribute.

If you are not a mod author then you don't need to worry about what this mod
does. Just install it as a dependency when requested.

## How to Use
1. Add a dependency on `KSPPluginLoader`.
   ```cs
   [assembly: KSPAssemblyDependency("KSPPluginLoader", 1, 0)]
   ```
   Dependencies on mods without a `KSPAssembly` attribute will not be resolved
   unless your mod DLL has this dependency.
2. Add `KSPAssemblyDependency` attributes for any other mods that you want to
   depend on. See the next section to pick the right version.

#### Figuring out the version to use
1. Open up `KSP.log` in your KSP directory.
2. Look up the `AssemblyLoader` log messages for your mod. They should look
   something like this:
   ```
   AssemblyLoader: Loading assembly at C:\KSP\GameData\ModuleManager.4.2.3.dll
   AssemblyLoader: KSPAssembly 'ModuleManager' V2.5.0
   ```
3. If there is a `KSPAssembly` message, then that's the version you should be
   using. If all your dependencies have a `KSPAssembly` message, then you
   don't actually need `KSPPluginLoader` (unless you're making use of the
   advanced use cases section below).
4. Otherwise, scroll on down to the section that starts with
   ```
   Mod DLLs found:
   ```
5. Find the entry for the mod you want to depend on. It should look something
   like this:
   ```
   KSPBurst v1.5.5.2
   Kopernicus v1.0.0.0 / v1.12.227.0
   ContractConfigurator v1.0.0.0 / v2.11.2.0 KSP-RO / v2.11.2.0
   ```
6. The version number you want is the _last_ version number after the mod
   name, since that is the version that `KSPPluginLoader` will use.
   You only need to care about the version number so you can ignore any text
   after the `+` in the version.


As an example, here's what the dependencies would look like for latest versions
of the three mods in step 5.
```cs
[assembly: KSPAssemblyDependency("KSPBurst", 1, 5, 5)]
[assembly: KSPAssemblyDependency("Kopernicus", 1, 12, 227)]
[assembly: KSPAssemblyDependency("ContractConfigurator", 2, 11, 2)]
```

## Advanced Use Cases
To support more advanced use cases, KSPPluginLoader also has some extra
assembly attributes you can use.

To use these you will need to actually depend on `KSPPluginLoader.dll`.
They also will not be checked unless your mod DLL has a `KSPAssemblyDependency`
on `KSPPluginLoader`.

### Maximum dependency version constraint
You might want to only load your mod if some other mod's version is within a
certain range. `KSPAssemblyDependency` only lets you set a lower bound, so
this mod introduces a `KSPAssemblyDependencyMax` which allows you to set an
upper bound.

Declaring it works pretty much the same as `KSPAssemblyDependency`.
This would only allow your mod to be loaded if `SomeMod`'s version is
< 1.0.0.
```cs
[assembly: KSPAssemblyDependencyMax("SomeMod", 1, 0, 0)]
```

As an example, the following two attributes would ensure your mod is only
loaded if a version of `PersistentThrust` in the range v1.7.5 <= v < v1.8.0
is present:
```cs
using KSPPluginLoader;

[assembly: KSPAssemblyDependency("PersistentThrust", 1, 7, 5)]
[assembly: KSPAssemblyDependencyMax("PersistentThrust", 1, 8, 0)]
```

In order for this attribute to make sure to:
* Add a `KSPAssemblyDependency` on `KSPPluginLoader`. Extra attributes like this
  are only checked if the DLL depends on `KSPPluginLoader`.
* Also add a `KSPAssemblyDependency` on the same mod as your
  `KSPAssemblyDependencyMax`. If you don't do this then then your DLL will never
  be loaded.

### Assembly Multiversioning
You can use assembly dependency constraints to load different mod DLLs
depending on a dependency version. The plugin loader will ensure only
one copy of any mod with a given name is loaded at a time, but you can use
dependency constraints to select which ones are considered.

The main use case for this is having multiple different implementations
for different KSP versions.

For example, here's what you would need to add in order for your mod DLL
to only be loaded for KSP v1.8:
```cs
[assembly: KSPAssemblyDependency("KSP", 1, 8)]
[assembly: KSPAssemblyDependencyMax("KSP", 1, 9)]
[assembly: KSPAssemblyDependency("KSPPluginLoader", 1, 0)]
```

## Examples
### Depending on CryoTanks
CryoTanks contains DLL called `SimpleBoiloff.dll` but that DLL does not
have a `KSPAssembly` attribute. With this mod, however, you can ignore
that by adding the following to your `AssemblyInfo.cs`:
```cs
[assembly: KSPAssemblyDependency("SimpleBoiloff", 0, 2, 1)]
[assembly: KSPAssemblyDependency("KSPPluginLoader", 1, 0)]
```
