﻿/*Copyright © 2016, wasml
 Licensed under the Attribution-ShareAlike 4.0 (CC BY-SA 4.0)
 creative commons license. See <https://creativecommons.org/licenses/by-nc-sa/4.0/>
 for full details.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
Scaling code adapted from TweakScale - 
   Licence DO WHAT THE FUCK YOU WANT TO PUBLIC LICENSE Copyright (C) 2004 Sam Hocevar
*/

using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using KSP.IO;

namespace Aerostat
{
    public class ModuleAerostat : PartModule
    {
        // LZ     Len   dia  Vol     Mass Kg  Lift Kg  L/V
        //   1    128   12    11,298       ?   12,428  1.10
        //  18    158   16    27,000  20,185   11,100  0.41 2.54 Kg/m2
        // 120    121   19    20,000  13,646    9,593  0.48 1.89Kg/m2
        // 127    236   30   105,000       ?   60,000  0.57
        // 130    245   41   200,000 130,000  232,000  1.16 4.12Kg/m2
        // Typical overpressure for non ridgid or semi ridgid 0.0045 - 0.0065 bar = 456 - 659 Pa
        // Blimps: The pressure inside the envelope is low, approximately 0.07 pounds per square inch (0.005 ATM).
        // 1 bar = 1.01325 atm = 100325 Pa

        // PV = NRT
        // P in pascals
        // V in liters
        // N in moles 0.04231453/liter at sea level, 288K
        // R = 8314.459848 L Kg mol K
        // T in kelvins

        /* ModuleRigidFrameAirship fields with default values
        float envelopeVolume = 1.0f;            // m3/s
        float maxFillRate = 20.0f;              // m3/s
        string fillGas = "Hydrogen";            // Cycle through choices
        float emergencyReliefValve = 0.3f;      // Atm over ambient pressure that the ERV will start venting gas
        float ERVRate = 10.0f;                  // m3/s released by the ERV
         */

        // And some other one per class stuff
        //private static double MAGIC_LIFT_MULTIPLIER = 0.009756;
        private static string modTag = "[Aerostat] ";
        private static int UNUSED = -1;
        public static int LIFTCELL = 0;
        public static int BALLOONET = 1;

        public static int SET_INIT = 0;
        public static int MAINTAIN = 1;

        //private static int meshIndex = 0;
        //private static int colliderIndex = 1;

        // GUI stuff
        //private UIPartActionWindow rightClickMenu;

        // Resize
        List<MeshFilter> MFL = new List<MeshFilter>();
        List<Transform> TFL = new List<Transform>();
        List<Component> CL = new List<Component>();
        
        //Vector3[] originalMesh;
        //Vector3[] originalCollider;
        //float originalVolume = 1.0f;
        //float originalMass = 0.0f;
        //float originalLength = 1.0f;
        //float originalDiameter = 1.0f;
        //Vector3 originalScale = new Vector3(1.0f, 1.0f, 1.0f);

        //Vector3[] modVerts;
        //float oldScaleDiameter, oldScaleLength = 1.0f;
        //public float stepMoles = 1000.0f;
        public float stepMoles = 1000.0f;
        public int gasCellIndex;
        public double lift;
        public float totalVolume = 0.0f;
        public float liftVolume = 0.0f;       // Volume of lift gas
        public float balloonetVolume = 0.0f;
        public ModuleGasCell[] gasCells = new ModuleGasCell[2];
        protected static Lib.IdealGasMath IGL = new Lib.IdealGasMath();

        // ------------------------------------------------------------------------------------------------

        [KSPField(isPersistant = true)]
        public byte envelopeID = 0;

        [KSPField(isPersistant = true)]
        public float defaultLength = 5.0f;

        [KSPField(isPersistant = true)]
        public float defaultDiameter = 5.0f;

        [KSPField(isPersistant = true)]
        public bool holdPressure = false;

        [KSPField(isPersistant = true)]
        public float targetVolume = 0.0f; // Lift volume in percent of total volume

        //[KSPField(guiActive = false, guiActiveEditor = true, guiName = "Ridgid", isPersistant = true)]
        public bool ridgid = true; // if liftCell under pressure and !ridgid then trouble

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "a1", isPersistant = false)]
        public float d1 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "a2", isPersistant = false)]
        public float d2 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "a3", isPersistant = false)]
        public float d3 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "a4", isPersistant = false)]
        public float d4 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = false, guiName = "a5", isPersistant = false)]
        public float d5 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = false, guiName = "a6", isPersistant = false)]
        public float d6 = 0.0f;

        //------------------------------------------------------------------------------------------------

        //[UI_FloatRange(controlEnabled = true, scene = UI_Scene.Editor, minValue = 0.1f, maxValue = 10.0f, stepIncrement = 0.1f)]
        //[KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "f1", guiName = "Length", isPersistant = true)]
        public float scaleLength = 1.0f;

        //[UI_FloatRange(controlEnabled = true, scene = UI_Scene.Editor, minValue = 0.1f, maxValue = 10.0f, stepIncrement = 0.1f)]
        //[KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "f1", guiName = "Diameter", isPersistant = true)]
        public float scaleDiameter = 1.0f;

        //------------------------------------------------------------------------------------------------

        public void SetVolumeHold(bool on)
        {
            holdPressure = on;
            if (holdPressure)
            {
                RecalcTargetVolume(true);
            }
        }

        // Cycles through the gas cells in the right click GUI
        [KSPEvent(active = true, guiActive = true, guiActiveEditor = true, guiName = "Cell Main")]
        public void CellIndexEvent()
        {
            gasCellIndex = (gasCellIndex + 1) % 2;
            Events["CellIndexEvent"].guiName = "Cell " + gasCells[gasCellIndex].cellName;
        }

        [KSPEvent(active = true, guiActive = true, guiName = "Step+ 1000")]
        public void StepUp()
        {
            stepMoles = stepMoles * 10;
            Events["StepUp"].guiName = "Step+ (" + stepMoles.ToString("F0") + ")";
        }

        [KSPEvent(active = true, guiActive = true, guiName = "Step-")]
        public void StepDown()
        {
            stepMoles = stepMoles * 0.1f;
            Events["StepUp"].guiName = "Step+ (" + stepMoles.ToString("F0") + ")";
        }

        [KSPEvent(active = true, guiActive = true, guiName = "Lift+", name = "Up")]
        public void IncLift()
        {
            try
            {
                gasCells[gasCellIndex].targetMoles += stepMoles;
                if (holdPressure)
                    RecalcTargetVolume(false);
            }
            catch
            { }
        }

        [KSPEvent(active = true, guiActive = true, guiName = "Lift-", name = "Down")]
        public void DecLift()
        {
            try
            {
                gasCells[gasCellIndex].targetMoles -= stepMoles;
                if (gasCells[gasCellIndex].targetMoles < 0)
                    gasCells[gasCellIndex].targetMoles = 0;
                if (holdPressure)
                    RecalcTargetVolume(false);
            }
            catch
            { }
        }

        //------------------------------------------------------------------------------------------------

        //private void SwapResource(int newResourceNum)
        //{
        //    //Debug.Log(modTag + "-> SwapResource(" + oldResourceNum.ToString() + "," + newResourceNum.ToString() + ")");

        //    // Save existing resources except for envelopeResource
        //    List<PartResource> saveResources = part.Resources.list;
        //    part.Resources.list.Clear();

        //    envelopeResource = AddResource(resourceInfo[gasIndex].name);
        //    //TODO: Need to fetch this on load too!

        //    for (int i = 0; i < saveResources.Count; i++)
        //        if (saveResources[i].info.id != envelopeResource.info.id)
        //            part.Resources.list.Add(saveResources[i]);

        //    //part.SendEvent("rightClick");

        //    //Debug.Log(modTag + "   SwapResource ->");
        //}

        //protected float GetLiftVolume()
        //{
        //    return gasCells[0].cellVolume;
        //}

        protected void RecalcTargetVolume(bool useCurrentMoles)
        {
            // Recalc volume ratio with targetMoles at current pressure
            double neededMoles = ModuleGasCell.CalcMoles(gasCells[LIFTCELL].targetPressure + gasCells[LIFTCELL].AmbientPressure_Pa(), gasCells[LIFTCELL].cellVolume, part.temperature);
            if (useCurrentMoles)
            {
                targetVolume = (float)(gasCells[LIFTCELL].gasMoles / neededMoles);
                gasCells[LIFTCELL].targetMoles = gasCells[LIFTCELL].gasMoles;
            }
            else
                targetVolume = (float)(gasCells[LIFTCELL].targetMoles / neededMoles);

            if (targetVolume < 0.0f) targetVolume = 0.0f;
            if (targetVolume > 1.0f) targetVolume = 1.0f;
        }

        public void RefreshRightClickWindow(Part part)
        {
            // Borrowed from TweakScale
            // foreach (var win in FindObjectsOfType<UIPartActionWindow>().Where(win => win.part == part))
            //    win.displayDirty = true;

            UIPartActionWindow[] RtClkMenu;
            RtClkMenu = FindObjectsOfType<UIPartActionWindow>();
            for (int i = 0; i < RtClkMenu.Length; i++)
                if (RtClkMenu[i].part == part)
                    RtClkMenu[i].displayDirty = true;
        }

        protected void ReSizePart(Vector3 newSize)
        {
            // Borrowed from TweakScale
            part.transform.GetChild(0).localScale = newSize;
            part.transform.GetChild(0).hasChanged = true;
            part.transform.hasChanged = true;
        }

        protected void AddLift()
        {
            //Debug.Log(modTag + "-> AddLift()");

            if (part.rigidbody != null)
            {
                Vector3 COL = part.rigidbody.worldCenterOfMass;
                double localg = FlightGlobals.getGeeForceAtPosition(COL).magnitude - Vector3.Dot(vessel.CoriolisAcc, vessel.upAxis);
                lift = 0.0;
                if (gasCells[LIFTCELL] != null)
                    lift += gasCells[LIFTCELL].gasVolume;
                if (gasCells[BALLOONET] != null)
                    lift += gasCells[BALLOONET].gasVolume;
                lift = lift * part.atmDensity * ModuleGasCell.KGMETERS3_TO_KGLITER * ModuleGasCell.liftMultiplier;

                // Why is this needed??? mass should be accounted for in vessel.GetTotalMass()
                if (gasCells[LIFTCELL] != null)
                    lift -= gasCells[LIFTCELL].gasMoles * ModuleGasCell.resourceInfo[gasCells[LIFTCELL].gasIndex].KgPerMole;
                if (gasCells[BALLOONET] != null)
                    lift -= gasCells[BALLOONET].gasMoles * ModuleGasCell.resourceInfo[gasCells[BALLOONET].gasIndex].KgPerMole;

                lift = lift * localg;
                Vector3 liftVec = vessel.upAxis * lift * ModuleGasCell.KG_TO_METRIC_TONS;
                part.rigidbody.AddForceAtPosition(liftVec, COL, ForceMode.Force);
            }

            //Debug.Log(modTag + "   AddLift ->");
        }

        protected void UpdateDragCubes()
        {
            //Debug.Log(modTag + "-> UpdateDragCubes");

            foreach (var dragCube in part.DragCubes.Cubes)
            {
                Vector3 DC;
                DC.x = DC.y = scaleDiameter;
                DC.z = scaleLength;
                dragCube.Size = DC;
                for (int i = 0; i < dragCube.Area.Length; i++)
                    dragCube.Area[i] = DC.y;
                for (int i = 0; i < dragCube.Depth.Length; i++)
                    dragCube.Depth[i] = DC.z;
            }

            part.DragCubes.ForceUpdate(true, true);

            //Debug.Log(modTag + "   UpdateDragCubes ->");
        }

        /// <summary>
        /// Rescales the envelope and recalculates mass and resource capacity
        /// Assumes resources scale as cube of size and mass as square of size (envelope shell)
        /// </summary>
        private void Rescale()
        {
            //Debug.Log(modTag + "-> Rescale");

            //part.mass = originalMass * scaleDiameter * scaleLength;

            //modVerts = MFL[0].mesh.vertices;
            //for (int j = 0; j < modVerts.GetLength(0); j++)
            //{
            //    modVerts[j].x = originalMesh[j].x * scaleDiameter;
            //    modVerts[j].y = originalMesh[j].y * scaleLength;
            //    modVerts[j].z = originalMesh[j].z * scaleDiameter;
            //}
            //MFL[0].mesh.vertices = modVerts;

            //modVerts = MFL[1].mesh.vertices;
            //for (int j = 0; j < modVerts.GetLength(0); j++)
            //{
            //    modVerts[j].x = originalCollider[j].x * scaleDiameter;
            //    modVerts[j].y = originalCollider[j].y * scaleLength;
            //    modVerts[j].z = originalCollider[j].z * scaleDiameter;
            //}
            //MFL[1].mesh.vertices = modVerts;

            ////part.attachNodes
            ////part.srfAttachNode

            //UpdateDragCubes();

            //for (int i = 0; i < gasCells.Count; i++)
            //{
            //    gasCells[i].cellVolume = originalVolume * scaleDiameter * scaleDiameter * scaleLength;
            //    gasCells[i].AddStorage(); // Updates the resource amount
            //}

            //RefreshRightClickWindow(part);

            //Debug.Log(modTag + "   Rescale ->");
        }

        public void HoldPressure()
        {
            if ((holdPressure) && (gasCells[LIFTCELL] != null))
            {
                double targetB = 0;
                double targetL = gasCells[LIFTCELL].targetMoles; // Default to current target moles
                double P = gasCells[LIFTCELL].targetPressure + gasCells[LIFTCELL].AmbientPressure_Pa();
                double T = part.temperature;
                // Calc moles for the entire envelope given volume, pressure and temperature
                double neededMoles = ModuleGasCell.CalcMoles(P, gasCells[LIFTCELL].cellVolume, T) - gasCells[LIFTCELL].targetMoles;
                double maxBalloonetMoles = ModuleGasCell.CalcMoles(P, gasCells[BALLOONET].cellVolume, T);
                if (neededMoles <= maxBalloonetMoles)
                {
                    if (neededMoles >= 0)
                        targetB = neededMoles;
                    else
                    {   // Over pressure and balloonet empty - remove lift gas
                        targetB = 0;
                        targetL = neededMoles + gasCells[LIFTCELL].gasMoles;
                    }
                }
                else
                {   // Unable to meet pressure goal with balloonet alone - add lift gas
                    targetB = maxBalloonetMoles;
                    targetL = neededMoles + gasCells[LIFTCELL].gasMoles - maxBalloonetMoles;
                }

                if (gasCells[LIFTCELL].gasLeak <= ModuleGasCell.NOT_A_LEAK)
                    gasCells[LIFTCELL].targetMoles = (float)targetL;
                else
                    holdPressure = false;
                if (gasCells[BALLOONET].gasLeak <= ModuleGasCell.NOT_A_LEAK)
                    gasCells[BALLOONET].targetMoles = (float)targetB;
                else
                    holdPressure = false;
            }
        }

        protected void CombineLiftEnvelopes()
        {
            // Distribute lift gas proportionatly between combined lift envelopes so Pa is the same in all
        }

        protected void RecalcPressure()
        {
            //Debug.Log(modTag + "-> RecalcPressure()");
            double AmbientPa = gasCells[BALLOONET].AmbientPressure_Pa();

            double pressure = Math.Max(gasCells[LIFTCELL].CalcPressure(gasCells[LIFTCELL].gasMoles + gasCells[BALLOONET].gasMoles, gasCells[LIFTCELL].cellVolume), AmbientPa);
            gasCells[BALLOONET].gasVolume = (float)(ModuleGasCell.calcVolume_L(gasCells[BALLOONET].gasMoles, part.temperature, pressure));
            if (gasCells[BALLOONET].gasVolume >= gasCells[BALLOONET].cellVolume)
            {   // Balloonet at full inflation
                gasCells[BALLOONET].gasVolume = gasCells[BALLOONET].cellVolume;
                gasCells[BALLOONET].pressureDifferential = (float)Math.Max(0, gasCells[BALLOONET].CalcPressure() - AmbientPa);
                d4 = gasCells[BALLOONET].pressureDifferential;
            }
            else
            {
                gasCells[BALLOONET].pressureDifferential = (float)Math.Max(0, pressure - AmbientPa);
            }

            gasCells[LIFTCELL].gasVolume = (float)gasCells[LIFTCELL].calcVolume_L(gasCells[LIFTCELL].gasMoles, gasCells[LIFTCELL].cellVolume - gasCells[BALLOONET].gasVolume);
            pressure = gasCells[LIFTCELL].CalcPressure();
            if (pressure < AmbientPa)
            {
                gasCells[LIFTCELL].gasVolume = (float)gasCells[LIFTCELL].calcVolume_L(gasCells[LIFTCELL].gasMoles, gasCells[LIFTCELL].gasVolume);
            }
            gasCells[LIFTCELL].pressureDifferential = (float)Math.Max(0, pressure - AmbientPa);

            //Debug.Log(modTag + "   RecalcPressure ->");
        }

        protected void Test()
        {
            
        }

        //------------------------------------------------------------------------------------------------

        public override void OnStart(StartState state)
        {
            //Debug.Log(modTag + "-> OnStart()");

            base.OnStart(state);

            //originalVolume = 0.0f;
            int index = LIFTCELL;
            for (int i = 0; i < part.Modules.Count; i++)
                if (part.Modules[i].moduleName == "ModuleGasCell")
                {
                    if (index <= BALLOONET)
                        gasCells[index] = (ModuleGasCell)part.Modules[i];
                    index++;
                }

            for (int i = index; i <= BALLOONET; i++)
            {
                gasCells[i] = new ModuleGasCell();
                if (gasCells[i] == null)
                    Debug.Log(modTag + "gasCells[" + i + "] == null");
            }

            // Swap so lift is in index 0 and balloonet is index 1
            if ((index >= BALLOONET) && (gasCells[LIFTCELL].gasIndex == ModuleGasCell.ATMOSPHERE))
            {
                ModuleGasCell temp = gasCells[LIFTCELL];
                gasCells[LIFTCELL] = gasCells[BALLOONET];
                gasCells[BALLOONET] = temp;
            }

            //Debug.Log(modTag + "LIFT:" + gasCells[LIFTCELL]);
            //Debug.Log(modTag + "BLLN:" + gasCells[BALLOONET]);

            //Debug.Log(modTag + "   OnStart ->");
        }

        public void FixedUpdate()
        {
            //Debug.Log(modTag + "-> FixedUpdate");

            if (HighLogic.LoadedSceneIsFlight)
            {
                AddLift();
                RecalcPressure();
            }
            else if (HighLogic.LoadedSceneIsEditor)
            {
                //if ((oldScaleLength != scaleLength) || (oldScaleDiameter != scaleDiameter))
                //{
                //    oldScaleLength = scaleLength;
                //    oldScaleDiameter = scaleDiameter;
                //    //Rescale();
                //}
            }

            //Debug.Log(modTag + "   FixedUpdate ->");
        }

    }
}