﻿/*Copyright © 2016, wasml
 Licensed under the Attribution-ShareAlike 4.0 (CC BY-SA 4.0)
 creative commons license. See <https://creativecommons.org/licenses/by-nc-sa/4.0/>
 for full details.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections.Generic;
using UnityEngine;
using KSP.IO;

namespace Aerostat
{
    public class ModuleGasCell : PartModule
    {
        public static float ATM_TO_PA = 101325.0f;   // 1 Atm in Pascals
        public static double R = 8314.459848; // Universal gas constant L, Pa, K, mol
        public static double LITER_PER_MOLE = 24.78959842; // At STP 24.78959842, 0deg 1 bar or 22.414 at 1 atm, 0deg
        public static double MOLES_PER_LITER = 0.0440315646118312; // At STP  0.04461497
        public static double STANDARD_TEMP = 273.15;      // K
        public static double STANDARD_PRESSURE = 100000.0;    //mB
        public static float KERBIN_ATM_DENSITY_STP = 0.001220f; // Kg/L
        public static double KG_TO_METRIC_TONS = 0.001;
        public static double METRIC_TONS_TO_KG = 1000;
        public static double METERS3_PER_LITER = 0.001;
        public static double KGMETERS3_TO_KGLITER = 0.001;
        public static double LITERS_PER_METERS3 = 1000.0;

        public static float NOT_A_LEAK = 0.01f;
        public const float MIN_ADJUST = 0.01f;
        public const double MIN_RESOURCE = 0.0000001;
        public const double PUMP_CHARGE_PER_MOLE = 0.00001;

        enum RESOURCES { VACUUM = 0, ATMOSPHERE, HYDROGEN, HELIUM };
        public const int VACUUM = 0;
        public const int ATMOSPHERE = 1;
        public const int HYDROGEN = 2; // 28.836 J/(mol·K) need KJ/Kt
        public const int HELIUM = 3; // 20.78[2] J/(mol·K)
        // Steam, Hot air
        public const int MIN_GAS = ATMOSPHERE;
        public const int MAX_GAS = HELIUM + 1;

        // And some other one per class stuff
        private static string modTag = "[GasCell] ";
        protected static System.Random rand = null;
        protected static int electricID = 0;

        //------------------------------------------------------------------------------------------------

        public struct ResourceInfo
        {
            public string name;
            public double KgPerLiter;
            public double KgPerMole; // Air:1.2754, H2:0.08988, He:0.1786
            public int id;
            public bool track; // f,f,f,t,t

            public void Setup(PartResourceDefinition resource)
            {
                id = resource.id;
                KgPerLiter = resource.density * METRIC_TONS_TO_KG;
                KgPerMole = KgPerLiter * LITER_PER_MOLE;
                name = resource.name;
                //Debug.Log(modTag + resource.name + ": " + resource.density * LITERS_PER_METERS3 + " Kg/m3, " + KgPerMole + " Kg/mol");
            }
        }

        protected static ResourceInfo infoInit = new ResourceInfo();

        public static ResourceInfo[] resourceInfo = new ResourceInfo[MAX_GAS]
        {
            new ResourceInfo{name="Vacuum",     KgPerLiter=0.00000000, KgPerMole=0.00000000, id=0, track=false},
            new ResourceInfo{name="LocalAtm",   KgPerLiter=0.00121000, KgPerMole=0.02895330, id=0, track=false},
            new ResourceInfo{name="H2",         KgPerLiter=0.00008988, KgPerMole=0.00201588, id=0, track=true },
            new ResourceInfo{name="He",         KgPerLiter=0.00017860, KgPerMole=0.00400260, id=0, track=true }
            //new ResourceInfo{name="Hydrogen",   KgPerLiter=0.00008988, KgPerMole=0.00201588, id=0, track=true },
            //new ResourceInfo{name="Helium",     KgPerLiter=0.00017860, KgPerMole=0.00400260, id=0, track=true }
          //new ResourceInfo{name="Ammonia",    KgPerLiterR=0.00076900,   KgPerMole=0.017031, id=0, track=true }
        };

        //------------------------------------------------------------------------------------------------

        protected float maxFillRateMoles = -1.0f; // 50 mol/s ~ 1m3/s
        public bool ERVVenting = false;

        private PartResource cellResource;

        // Only load configs, resources once
        private static bool resourcesLoaded = false;
        private static bool configLoaded = false;

        //private static bool Init = false;
        float deltaTime;

        //---------------------------------------- Fields ------------------------------------------------

        [KSPField(guiActive = true, guiActiveEditor = true, guiName = "Name", isPersistant = true)]
        public string cellName = "Gas Cell";

        [KSPField(isPersistant = true)]
        public byte envelopeID = 0;

        [KSPField(isPersistant = true)]
        public int gasIndex = HYDROGEN;   // Vacuum, Atm, (H), He

        // This is the total volume of the envelope in m³
        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "F1", guiName = "Volume", guiUnits = " Liters", isPersistant = true)]
        public float cellVolume = 1.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "F0", guiName = "Fill Rate", guiUnits = " l/s", isPersistant = true)]
        public float maxFillRate = 2000.0f;

        // Pressure differential in Pa
        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F0", guiName = "Max Pa", isPersistant = true)]
        public float maxPressure = 1500.0f;

        [KSPField(isPersistant = true)]
        public float targetPressure = 500.0f;   // Aim for target Pa unless Vol hold set - then keep Pa in limits

        [KSPField(isPersistant = true)]
        public float minTgtPa = 450.0f;         // Lower limit (Pa)

        [KSPField(isPersistant = true)]
        public float maxTgtPa = 550.0f;         // Upper limit (Pa)

        [KSPField(isPersistant = true)]         // Pa
        public float ERVPressure = 1425.0f;     // 95% max

        // Vent rate in m3 (STP)
        [KSPField(isPersistant = true)]
        public float ERVRate = 50.0f;

        [KSPField(isPersistant = true)]
        public bool venting = false;

        [KSPField(isPersistant = true)]
        public float ventRate = 50.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiName = "Overfill Prot", isPersistant = true)]
        public bool overFillProtect = true;

        [KSPField(isPersistant = true)]
        public float gasLeak = 0.0f;

        [KSPField(guiActive = false, guiActiveEditor = false, guiFormat = "F2", guiName = "Leak", guiUnits = " l/s", isPersistant = true)]
        public float gasLoss = 0.0f;

        [KSPEvent(active = true, guiActive = true, guiActiveEditor = false, guiName = "Main Dump Off")]
        public void toggleDumpEvent()
        {
            venting = !venting;
            if (venting)
                Events["toggleDumpEvent"].guiName = cellName + " Dump " + ERVRate.ToString("F0") + " L/s";
            else
                Events["toggleDumpEvent"].guiName = cellName + " Dump Off";
        }

        //[KSPEvent(active = true, guiActiveEditor = true, guiName = "Lift Gas: Hydrogen")]
        public void changeGas()
        {
            gasIndex = (gasIndex + 1) % MAX_GAS;
            Events["changeGas"].guiName = "Lift Gas: " + resourceInfo[gasIndex].name;

            // This should be part of resourceInfo
            if (gasIndex == HYDROGEN)
                part.explosionPotential = 0.1f;
            else
                part.explosionPotential = 0.0f;

            //SwapResource(gasIndex);

            //Events["IncLift"].guiActive = (gasIndex != ATMOSPHERE);
            //Events["IncLift"].active = (gasIndex != ATMOSPHERE);
            //Events["DecLift"].guiActive = (gasIndex != ATMOSPHERE);
            //Events["DecLift"].active = (gasIndex != ATMOSPHERE);

            //Events["IncBallonet"].guiActive = (gasIndex != VACUUM);
            //Events["IncBallonet"].active = (gasIndex != VACUUM);
            //Events["DecBallonet"].guiActive = (gasIndex != VACUUM);
            //Events["DecBallonet"].active = (gasIndex != VACUUM);
        }

        [KSPField(guiActive = false, guiActiveEditor = false, guiFormat = "f1", guiName = "Lift Mult ", isPersistant = false)]
        public static float liftMultiplier = 10.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g1", isPersistant = false)]
        public float d1 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g2", isPersistant = false)]
        public float d2 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g3", isPersistant = false)]
        public float d3 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g4", isPersistant = false)]
        public float d4 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g5", isPersistant = false)]
        public float d5 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "F4", guiName = "g6", isPersistant = false)]
        public float d6 = -2.0f;

        //[KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "F0", guiName = "Tgt Moles", isPersistant = true)]
        public float targetMoles = 0.0f; // Target moles fill gas

        // Tracks the current moles of gas in the cell
        [KSPField(guiActive = false, guiFormat = "F1", guiName = "Moles", isPersistant = true)]
        public float gasMoles = 0.0f;

        // Tracks the current volume of gas in the cell
        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "F2", guiName = "Vol", guiUnits = " liters", isPersistant = false)]
        public float gasVolume = 0.0f;

        // Cell overpressure in Pa
        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "F1", guiName = "Pressure", guiUnits = " Pa", isPersistant = false)]
        public float pressureDifferential = 0.0f;

        [KSPEvent(active = true, externalToEVAOnly = true, unfocusedRange = 100, guiActiveUnfocused = true, guiActive = true, guiActiveEditor = false, guiName = "Repair")]
        public void Repair()
        {
            // How do we make it so only an engineer can do this?
            gasLeak = 0.0f;
            Events["Repair"].active = false;
            Fields["gasLeak"].guiActive = false;
        }

        [KSPField(isPersistant = true)]
        public float baseLength = 5.0f;

        [KSPField(isPersistant = true)]
        public float baseRadius = 5.0f;

        //------------------------------------------------------------------------------------------------

        public float SetGasMoles(double newAmount)
        {
            float rtn = 0.0f;

            if (cellResource != null)
            {
                cellResource.amount = Math.Max(0, newAmount);
                rtn = gasMoles = (float)(newAmount * MOLES_PER_LITER);
            }
            return rtn;
        }

        //------------------------------------------------------------------------------------------------
        // All these funtions operate in Kg, Liters, Pa and moles unless stated otherwise

        // Not used?
        public float GetAtmDensityFromEditor(Vector3d Pos)
        {
            double altitude = vessel.mainBody.GetAltitude(Pos);
            double P = vessel.mainBody.GetPressure(altitude);
            double T = vessel.mainBody.GetTemperature(altitude);
            return (float)vessel.mainBody.GetDensity(P, T);
        }

        public static double molesToSTPVolume(double N)
        {
            //Debug.Log(modTag + "STPVolume(N)");
            return N * R * STANDARD_TEMP / STANDARD_PRESSURE;
        }

        public static double calcVolume_L(double N, double T, double P)
        {
            //Debug.Log(modTag + "calcVolume(N, T, P)");
            try
            { return N * R * T / P; }
            catch
            { return 0.0; }
        }

        public double calcSTPVolume_L(double Volume)
        {
            return (Volume * MOLES_PER_LITER * R * part.temperature) / AmbientPressure_Pa();
        }

        protected double calcUnrestrainedVolume_L()
        {
            //Debug.Log(modTag + "calcVolume(STPV)");
            if (cellResource == null)
                return 0.0f;

            return (cellResource.amount * MOLES_PER_LITER * R * part.temperature) / AmbientPressure_Pa();
        }

        public double calcVolume_L(double molesGas, double maxVolume)
        {
            //Debug.Log(modTag + "calcVolume(molesGas)");
            double rtn = 0.0;

            try
            {   // V = NRT/P
                rtn = (molesGas * R * part.temperature) / AmbientPressure_Pa();
                if ((rtn > maxVolume) && (maxVolume >= 0))
                    rtn = maxVolume;
            }
            catch
            { rtn = 0.0; }

            return rtn;
        }

        public static double CalcMoles(double P, double V, double T)
        {
            //Debug.Log(modTag + "calcMoles(P, V, T)");
            try
            { return (P * V) / (R * T); }
            catch
            { return 0.0; }
        }

        public double CalcMoles(double V) // V in liters
        {
            //Debug.Log(modTag + "calcMoles(V)");
            try
            { return (AmbientPressure_Pa() * V) / (R * part.temperature); }
            catch
            { return 0.0; }
        }

        // Returns the ambient pressure in Pa
        public double AmbientPressure_Pa()
        {
            //Debug.Log(modTag + "AmbientPressure_Pa()");
            if (part != null)
                return part.staticPressureAtm * ATM_TO_PA;
            else
                return 0.0;
        }

        public double AmbientDensity_L()
        {
            if (part != null)
                return part.atmDensity / METERS3_PER_LITER;
            else
                return 0.0;
        }

        public double CalcPressure()
        {
            //Debug.Log(modTag + "calcPressure()");
            if (gasVolume > 0.001)
            {
                // P = NRT/V
                return (gasMoles * R * part.temperature / gasVolume);
            }
            else
                return 0.0;
        }

        public double CalcPressure(double N)
        {
            //Debug.Log(modTag + "calcPressure(" + N.ToString() + ")");
            /* This is only valid if the envelope is full */
            if (cellVolume > 0.01)
            {
                // P = NRT/V
                return (N * R * part.temperature / cellVolume);
            }
            else
                return 0.0;
        }

        public double CalcPressure(double N, double V)
        {
            //Debug.Log(modTag + "calcPressure(" + N + "," + V + ")");
            if (V > 0.001)
            {
                // P = NRT/V
                return (N * R * part.temperature / V);
            }
            else
                return 0.0;
        }

        public double CalcMolesForPressure(double targetPressure)
        {
            return (targetPressure * cellVolume) / (R * part.temperature);
        }

        //------------------------------------------------------------------------------------------------

        public ModuleGasCell()
        {
            gasIndex = -1;
            gasMoles = 0;
            gasVolume = 0;
            cellVolume = 0;
            pressureDifferential = 0;
        }

        public void SwapResource(int newResourceNum)
        {
            //Debug.Log(modTag + "-> SwapResource(" + oldResourceNum.ToString() + "," + newResourceNum.ToString() + ")");

            part.Resources.list.Remove(cellResource);
            cellResource = AddResourceToPart(part, resourceInfo[newResourceNum].name, cellVolume);
            part.Resources.UpdateList();

            // Save existing resources except for envelopeResource
            //List<PartResource> saveResources = part.Resources.list;
            //part.Resources.list.Clear();

            //cellResource = AddResource(resourceInfo[gasIndex].name);
            //TODO: Need to fetch this on load too!

            //for (int i = 0; i < saveResources.Count; i++)
            //    if (saveResources[i].info.id != cellResource.info.id)
            //        part.Resources.list.Add(saveResources[i]);

            //Debug.Log(modTag + "   SwapResource ->");
        }

        public void RemoveResource(int id)
        {
            //Debug.Log(modTag + "-> RemoveResource()");

            // Add the resource storage to the part
            
            cellResource = part.Resources.Get(id);
            if (cellResource != null)
            {
                part.Resources.list.Remove(cellResource);
                cellResource = null;
            }

            //Debug.Log(modTag + "   RemoveResource() ->");
        }

        public void AddStorage()
        {
            //Debug.Log(modTag + "-> AddStorage()");

            RemoveResource(resourceInfo[gasIndex].id);
            cellResource = AddResourceToPart(part, resourceInfo[gasIndex].name, cellVolume);

            //Debug.Log(modTag + "   AddStorage() ->");
        }

        protected double MoveGas(double N)
        {
            // This method isolates the KSP STP storage volume units
            // N>0 is in and N<0 is out

            // Convert to m3
            double Liters = N * LITER_PER_MOLE;
            if (cellResource != null)
            {
                cellResource.flowState = true;
                Liters = part.RequestResource(cellResource.resourceName, Liters);
                SetGasMoles(cellResource.amount + Liters);
                cellResource.flowState = false; // Lock so envelope isn't used as storage
            }
            else
            {
                DebugUtil.LogUtil.Log(modTag + "MoveGas: cellResource == null");
            }

            return Liters * MOLES_PER_LITER; // Return in moles moved
        }

        public int FindResourceID(string st)
        {
            PartResourceDefinition resource = PartResourceLibrary.Instance.GetDefinition(st);
            return resource.id;
        }

        protected float PumpResouce(float moles)
        {
            // Note: We pump in/out m3 (based on STP) but convert to moles for use. 
            //Debug.Log(modTag + "-> pumpResouce:" + moles.ToString());

            if (electricID == 0)
            {
                electricID = FindResourceID("ElectricCharge");
                //PartResourceDefinition resource = PartResourceLibrary.Instance.GetDefinition("ElectricCharge");
                //electricID = resource.id;
                if (electricID == 0)
                {
                    DebugUtil.LogUtil.Log(modTag + "PumpResouce: EC not found");
                    return 0;
                }
            }

            float drawResource = 0.0f;

            // Draw electric charge to run pump if needed
            double requiredElectric = Math.Abs(moles) * PUMP_CHARGE_PER_MOLE * Time.fixedDeltaTime;
            if (requiredElectric > MIN_RESOURCE)
            {
                drawResource = (float)part.RequestResource(electricID, requiredElectric);
                if (drawResource < requiredElectric)
                    moles = (float)(moles * (drawResource / requiredElectric));
            }

            moles = (float)MoveGas(moles);

            //Debug.Log(modTag + "   pumpResouce -> " + moles.ToString());
            return moles;
        }

        public void Leak()
        {
            if (cellResource == null)
            {
                DebugUtil.LogUtil.Log(modTag + "CheckLeak: cellResource == null ("+ cellName + ")");
                return;
            }

            if (gasLeak > NOT_A_LEAK)
            {
                gasLoss = gasLeak + (pressureDifferential * pressureDifferential * 0.1f); // pressure pushing gas out
                if ((gasIndex != ATMOSPHERE) && (gasVolume > cellVolume * 0.5)) // Assumes leak at half way point and no mixing (or tilting)
                    gasLoss += (float)(gasLeak * 0.5);     // Lift gas exiting
                gasLoss += (float)(gasLeak * 0.05);        // Weight of envelope pushing gas out

                SetGasMoles(cellResource.amount - deltaTime * gasLoss);
            }

            //Debug.Log(modTag + "Leaking");
        }

        protected void EmergencyReliefValve()
        {
            //Debug.Log(modTag + "EmergencyReliefValve");
            if (cellResource != null)
            {
                ERVVenting = pressureDifferential >= 1425;
                if (ERVVenting)
                {
                    float ERVAmt = (float)(deltaTime * ERVRate * MOLES_PER_LITER * pressureDifferential * 0.0001);
                    SetGasMoles(Math.Max(cellResource.amount - ERVAmt, 0));
                    // Display "Venting"
                    // Sound warning?
                }
            }

            //Debug.Log(modTag + "Exit EmergencyReliefValve");
        }

        protected void OverFillProtection()
        {
            // If we're close to venting don't allow adding any more
            if (overFillProtect && (pressureDifferential >= 1425 * 0.90))
            {
                //targetMoles = (float)Math.Min(gasMoles, targetMoles);
                // Highlight overfill prot
            }
        }

        protected void AdjustFill()
        {
            //Debug.Log(modTag + "-> AdjustFill()");
            if (cellResource == null)
            {
                DebugUtil.LogUtil.Log(modTag + "AdjustFill: cellResource == null");
                return;
            }

            // Bail if we're too far off ideal pressure
            float deltaMoles = (float)(targetMoles - gasMoles);
            if (((deltaMoles > 0) && (pressureDifferential > maxTgtPa)) ||
                ((deltaMoles < 0) && (pressureDifferential < minTgtPa) && 
                 (!vessel.Landed) && (!vessel.Splashed) && (gasLeak <= NOT_A_LEAK)))
                return;

            // If we're close, set the target to the current fill
            // prevents adding lift unexpectedly when the ballonet is emptied
            if (Math.Abs(deltaMoles) < MIN_ADJUST)
            {
                targetMoles = gasMoles;
            }
            else
            {
                // Don't remove more gas than is present
                if (gasMoles + deltaMoles < 0)
                    deltaMoles = (float)-gasMoles;

                // Convert maxFillRate from liters/s to mol/s. N = PV/RT
                if (part.temperature > 0.01)
                    maxFillRateMoles = (float)((AmbientPressure_Pa() * maxFillRate) / (R * part.temperature));

                // Limit to maxFillRate
                deltaMoles = Mathf.Clamp(deltaMoles, -maxFillRateMoles, maxFillRateMoles);

                deltaMoles = deltaMoles * Time.fixedDeltaTime;
                deltaMoles = PumpResouce(deltaMoles);
                // If no resource found - stop.
                if (deltaMoles == 0.0f)
                {
                    targetMoles = gasMoles;
                }
            }

            SetGasMoles(cellResource.amount - gasLeak * Time.deltaTime);

            if (venting)
            {
                SetGasMoles(cellResource.amount - ventRate * Time.deltaTime);
            }

            //Debug.Log(modTag + "   AdjustFill ->");
        }

        protected void CheckRupture(ref float gasLeak, float moles)
        {
            //Debug.Log(modTag + "-> RuptureCheck");

            if (pressureDifferential > 1500)
            {
                //double overPressure = (pressureDifferential - maxPressure) / maxPressure;
                double overPressure = (pressureDifferential - 1500) / 1500;
                // Don't want lift and bolloonet bladder to rupture at the same time so have a random element dependent on over pressure
                if (rand.NextDouble() < deltaTime * 0.1 * overPressure * overPressure * overPressure)
                {
                    gasLeak += (float)(2 * overPressure * overPressure * overPressure + rand.NextDouble());
                    Fields["gasLeak"].guiActive = true;
                    targetMoles = Math.Min(gasMoles, targetMoles);
                    Events["Repair"].active = true;
                }
                // How do we make this repairable?
            }

            //Debug.Log(modTag + "   RuptureCheck ->");
        }

        public float Lift()
        {
            return (float)(gasVolume * part.atmDensity * KGMETERS3_TO_KGLITER * liftMultiplier);
        }

        //------------------------------------------------------------------------------------------------

        protected static void CreateResource(string name, float density, int hsp, string transfer, bool isTweakable, float unitCost)
        {
            //Debug.Log(modTag + "-> CreateResource(" + name + ", " + density + ")");

            ConfigNode newConfigNode = new ConfigNode("RESOURCE_DEFINITION");
            newConfigNode.AddValue("name", name);
            newConfigNode.AddValue("density", density); // in Kt/m3 = Kg/L
            if (hsp > 0)
                newConfigNode.AddValue("hsp", hsp);
            if (transfer != "")
                newConfigNode.AddValue("transfer", transfer);
            newConfigNode.AddValue("isTweakable", isTweakable);
            if (unitCost >= 0)
                newConfigNode.AddValue("unitCost", unitCost);
            bool OK = HighLogic.CurrentGame.config.AddNode(newConfigNode) != null;
            if (OK)
                Debug.Log(modTag + name + " created");
            else
                Debug.Log(modTag + "CreateResource: Unable to create " + name);

            //Debug.Log(modTag + "   CreateResource -> " + OK);
        }

        protected static void DestroyResource(string name)
        {
            //Debug.Log(modTag + "-> DestroyResource(" + name + ")");

            //bool OK = true;
            ConfigNode OldConfigNode = HighLogic.CurrentGame.config.GetNode("RESOURCE_DEFINITION", "name", name);
            if (OldConfigNode != null)
            {
                HighLogic.CurrentGame.config.RemoveNode(OldConfigNode);
                //Debug.Log(modTag + name + " removed");
            }
            else
            {
                Debug.Log(modTag + name + " not found");
                //OK = false;
            }

            //Debug.Log(modTag + "   DestroyResource -> " + OK);
        }

        public static PartResource AddResourceToPart(Part p, string name, float MaxResource)
        {
            //Debug.Log(modTag + "-> AddResourceToPart(" + name + ")");

            ConfigNode newConfigNode = new ConfigNode("RESOURCE");
            newConfigNode.AddValue("name", name);
            newConfigNode.AddValue("maxAmount", MaxResource.ToString());
            newConfigNode.AddValue("amount", 0.0f);
            newConfigNode.AddValue("flowState", "false");
            newConfigNode.AddValue("hideFlow", "false");

            //Debug.Log(modTag + "   AddResourceToPart -> ");
            return p.AddResource(newConfigNode);
        }

        protected static bool LoadGasInfo(int Num)
        {
            Debug.Log(modTag + "-> LoadGasInfo(" + Num + "): " + resourceInfo[Num].name);

            PartResourceDefinition resource = PartResourceLibrary.Instance.GetDefinition(resourceInfo[Num].name);
            bool loadOK = resource != null;
            if (loadOK)
            {
                resourceInfo[Num].Setup(resource);
                DebugUtil.LogUtil.Log(modTag + resourceInfo[Num].name + " set up");
            }
            else
            {
                DebugUtil.LogUtil.Log(modTag + resourceInfo[Num].name + " not found");
            }

            Debug.Log(modTag + "   LoadGasInfo ->");
            return loadOK;
        }

        private static void LoadConstants()
        {
            //Debug.Log(modTag + "-> LoadConstants");

            if (!configLoaded)
            {
                LoadConstants();

                // A random number gen in case we need to pop a gas bladder
                rand = new System.Random();
                configLoaded = true;
            }

            // Should realy do this in the save file with an options menu
            PluginConfiguration config = PluginConfiguration.CreateForType<ModuleGasCell>();
            config.load();
            liftMultiplier = config.GetValue<float>("cheatFactor", 1.0f);

            //Debug.Log(modTag + "   LoadConstants ->");
        }

        //---------------------------------------------------------------------------------

        protected void LoadResources()
        // Create resources for this mod.
        // LocalAtm density will vary from planet to planet and we don't want to change density on resources other mods may be using.
        {
            //DebugUtil.LogUtil.Log(modTag + "-> LoadResources");

            if (PartResourceLibrary.Instance == null)
                DebugUtil.LogUtil.Log(modTag + "PartResourceLibrary.Instance == null");
            else
            {
                // Remove LocalAtm and recreate with local planet values
                float atmDesity;
                DestroyResource(resourceInfo[ATMOSPHERE].name);
                if (HighLogic.LoadedSceneIsFlight)
                {
                    atmDesity = (float)vessel.mainBody.atmDensityASL;
                }
                else
                {
                    atmDesity = (float)FlightGlobals.GetHomeBody().atmDensityASL;
                }
                CreateResource(resourceInfo[ATMOSPHERE].name, (float)(atmDesity * METERS3_PER_LITER * KG_TO_METRIC_TONS), 1, "PUMP", true, 0.0f);
                LoadGasInfo(ATMOSPHERE);

                PartResourceDefinition resource = PartResourceLibrary.Instance.GetDefinition(resourceInfo[HYDROGEN].name);
                if (resource == null)
                    CreateResource(resourceInfo[HYDROGEN].name, 0.00000008988f, 1, "PUMP", true, 0.00009f);
                LoadGasInfo(HYDROGEN);

                resource = PartResourceLibrary.Instance.GetDefinition(resourceInfo[HELIUM].name);
                if (resource == null)
                    CreateResource(resourceInfo[HELIUM].name, 0.00000017860f, 1, "PUMP", true, 0.00249f);

                LoadGasInfo(HELIUM);

                resourcesLoaded = true;
            }

            //DebugUtil.LogUtil.Log(modTag + "    LoadResources ->");
        }

        public override void OnStart(StartState state)
        {
            //Debug.Log(modTag + "-> OnStart()");

            base.OnStart(state);

            if (cellVolume < 0.01)
                cellVolume = 0.01f;

            liftMultiplier = AerostatScenario.LiftMultiplier;

            if (rand == null)
                rand = new System.Random();

            Events["toggleDumpEvent"].guiActive = gasIndex != ATMOSPHERE;

            if (!resourcesLoaded)
                LoadResources();

            cellResource = part.Resources.Get(resourceInfo[gasIndex].id);
            if (cellResource == null)
            {
                cellResource = AddResourceToPart(part, resourceInfo[gasIndex].name, cellVolume);
                //Debug.Log(modTag + "Added " + resourceInfo[gasIndex].name + " to part in OnStatrt");
            }

            if (cellResource != null)
            {
                SetGasMoles(cellResource.amount);
            }
            else
                Debug.Log(modTag + "cellResource not found");

            // Toggle twice to update cell name and return to original state
            toggleDumpEvent();
            toggleDumpEvent();

            //Debug.Log(modTag + "   OnStart ->");
        }

        public void FixedUpdate()
        {
            //Debug.Log(modTag + "-> FixedUpdate");

            if (HighLogic.LoadedSceneIsFlight)
            {
                deltaTime = Time.fixedDeltaTime;
                Leak();
                if (!FlightGlobals.ActiveVessel.HoldPhysics && (pressureDifferential > 0.0f)) // FlightGlobals.ready  FlightGlobals.ActiveVessel.HoldPhysics
                {
                    OverFillProtection();
                    EmergencyReliefValve();
                    CheckRupture(ref gasLeak, gasMoles);
                }
                AdjustFill();
            }
            else if (HighLogic.LoadedSceneIsEditor)
            {
                if (cellResource != null)
                {
                    SetGasMoles(cellResource.amount);
                    targetMoles = gasMoles;
                }
            }

            //Debug.Log(modTag + "   FixedUpdate ->");
        }

    }
}
