﻿/*Copyright © 2016, wasml
 Licensed under the Attribution-ShareAlike 4.0 (CC BY-SA 4.0)
 creative commons license. See <https://creativecommons.org/licenses/by-nc-sa/4.0/>
 for full details.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections.Generic;
using UnityEngine;
using KSP.IO;

namespace Aerostat
{
    public class ModuleAerostatControler : PartModule
    {
        private static string modTag = "[AirshipControler] ";
        private static double RadToDeg = 180 / Math.PI;

        public float totalMoles = 0.0f;
        private float lastRate = 0.0f;
        private double lastAltitude = 0.0f;
        private PartResource LocalAtm = null;
        private AerostatControlGUI GUIWindow = null;
        protected List<ModuleGasCell> cellList = new List<ModuleGasCell>();
        protected List<ModuleAerostat> envelopeList = new List<ModuleAerostat>();
        private float NextUpdate = 0.0f;

        // Used to calc stats for display in editor - lift, Max alt, etc.
        protected List<CelestialBody> bodiesWithAtmosphere = new List<CelestialBody>();
        protected int currentBodyIndex = -1;        // Index into bodiesWithAtmosphere. -1 is unset.
        protected CelestialBody selectedBody = null; // Currently seleted body from bodiesWithAtmosphere

        //----------------------------------------- Fields -----------------------------------------------//

        [KSPField(isPersistant = true)]
        bool preventVenting = false;

        [KSPField(isPersistant = true)]
        bool MaintainTrim = false;

        [KSPField(isPersistant = true)]
        bool MaintainPressure = false;

        [KSPField(isPersistant = true)]
        private bool holdAltitude = false;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N0", guiName = "Min H2", guiUnits = " liters", isPersistant = false)]
        public float minH2 = 0.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N0", guiName = "Cur H2", guiUnits = " liters", isPersistant = false)]
        public float curH2 = 0.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N0", guiName = "Max H2", guiUnits = " liters", isPersistant = false)]
        public float maxH2 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "N1", guiName = "Volume", guiUnits = " liters", isPersistant = false)]
        protected float totalVolume = 0.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N0", guiName = "Max Alt", guiUnits = " m", isPersistant = false)]
        public float maxAlt = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "F1", guiName = "Temp Diff", guiUnits = " K", isPersistant = false)]
        public float TempDiff = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "f1", guiName = "Trim", guiUnits = " Deg", isPersistant = false)]
        public float trim = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "+0.000;-0.000; 0.000", guiName = "Vert Rate", guiUnits = " m/s", isPersistant = false)]
        public float rate = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "c1", isPersistant = false)]
        public float d1 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "c2", isPersistant = false)]
        public float d2 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "c3", isPersistant = false)]
        public float d3 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "c4", isPersistant = false)]
        public float d4 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiName = "c4", isPersistant = false)]
        public float d5 = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "F0", guiName = "Altitude Set", isPersistant = true)]
        //        private double altitudeSet = 2000.0f;
        private double altitudeSet = 70.0f;

        [KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "N1", guiName = "Mass", guiUnits = " Kg", isPersistant = false)]
        public float totalMass = 0.0f;

        //[KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "N1", guiName = "Lift", guiUnits = " tons", isPersistant = false)]
        public float CurLift = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = true, guiFormat = "N1", guiName = "Buoyancy", guiUnits = " N", isPersistant = false)]
        public float totalBuoyancy = 0.0f;

        [KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N1", guiName = "Max Bouy", guiUnits = " N", isPersistant = false)]
        public float maxLift = 0.0f;

        //[KSPField(guiActive = false, guiActiveEditor = true, guiFormat = "N1", guiName = "Lift Gas", guiUnits = " liters(STP)", isPersistant = false)]
        public float totalLiftGas = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "N0", guiName = "Max Pressure", guiUnits = " Pa", isPersistant = false)]
        public float maxPressureDiff = 0.0f;

        [KSPField(guiActive = true, guiActiveEditor = false, guiFormat = "N0", guiName = "Min Pressure", guiUnits = " Pa", isPersistant = false)]
        public float minPressureDiff = 0.0f;

        [KSPEvent(active = true, guiActive = true, guiActiveEditor = true, guiName = "Pressure Hold Off")]
        public void toggleMaintainPressure()
        {
            SetMaintainPressure(!MaintainPressure);
        }

        //[KSPEvent(active = true, guiActive = true, guiActiveEditor = false, guiName = "Trim Hold Off")]
        public void toggleMaintainTrim()
        {
            MaintainTrim = !MaintainTrim;
            if (MaintainTrim)
                Events["toggleMaintainTrim"].guiName = "Trim Hold Off";
            else
                Events["toggleMaintainTrim"].guiName = "Trim Hold On";
        }

        [KSPEvent(active = true, guiActive = false, guiActiveEditor = true, guiName = "Body()")]
        public void toggleBody()
        {
            if (bodiesWithAtmosphere.Count > 0)
            {
                currentBodyIndex++;
                if (currentBodyIndex >= bodiesWithAtmosphere.Count)
                    currentBodyIndex = 0;
                selectedBody = bodiesWithAtmosphere[currentBodyIndex];
                Events["toggleBody"].guiName = "Body(" + selectedBody.name + ")";
                SetAtmDensity();
            }
        }

        [KSPEvent(active = true, guiActive = true, guiActiveEditor = true, guiName = "Venting On")]
        public void togglePreventVenting()
        {
            preventVenting = !preventVenting;
            if (preventVenting)
                Events["togglePreventVenting"].guiName = "Venting Off";
            else
                Events["togglePreventVenting"].guiName = "Venting On";
        }

        //[KSPEvent(active = true, guiActive = true, guiActiveEditor = true, guiName = "GUI")]
        public void toggleGUI()
        {
            GUIWindow.guiVisible = !GUIWindow.guiVisible;
            GUIWindow.guiVisible = true;
        }

        //[KSPEvent(active = true, guiActive = true, guiActiveEditor = false, guiName = "Altitude Hold Off")]
        public void toggleAltitudeEvent()
        {
            //Debug.Log(modTag + "-> toggleAltitudeEvent");

            holdAltitude = !holdAltitude;
            ChangeAltitudeSetting();
            //Events["toggleAltitudeEvent"].guiActive = holdAltitude;
            //Events["altitudeSet"].guiActive = holdAltitude;
            Events["AltitudePlusEvent"].guiActive = holdAltitude;
            Events["AltitudeMinusEvent"].guiActive = holdAltitude;

            altitudeSet = vessel.altitude;

            //Debug.Log("[AirshipControler] toggleAltitudeEvent ->");
        }

        //[KSPEvent(active = true, guiActive = false, guiActiveEditor = false, guiName = "Altitude+")]
        public void AltitudePlusEvent()
        {
            altitudeSet += 100;
            ChangeAltitudeSetting();
            Events["toggleAltitudeEvent"].guiActive = true;
        }

        //[KSPEvent(active = true, guiActive = false, guiActiveEditor = false, guiName = "Altitude-")]
        public void AltitudeMinusEvent()
        {
            altitudeSet -= 100;
            ChangeAltitudeSetting();
            Events["toggleAltitudeEvent"].guiActive = true;
        }

        //----------------------------------------- Methods ----------------------------------------------//
        protected void ChangeAltitudeSetting()
        {
            if (holdAltitude)
            {
                SetForAltitude(altitudeSet);
                Events["toggleAltitudeEvent"].guiName = "Altitude Hold " + altitudeSet.ToString("F0") + " m";
            }
            else
            {
                Events["toggleAltitudeEvent"].guiName = "Altitude Hold Off";
            }
        }

        protected void GetSpawnPointPT(ref double P, ref double T)
        {
            P = 0;
            T = 0;
            if ((HighLogic.LoadedScene == GameScenes.EDITOR) && (EditorLogic.RootPart != null))
            {
                double alt = 0.0;

                // Get conditions at the spawn point (P, T)
                PSystemSetup.SpaceCenterFacility.SpawnPoint[] SpawnPoints = PSystemSetup.Instance.GetSpaceCenterFacility(EditorLogic.fetch.launchSiteName).spawnPoints;

                // Note - this retrieves the reference temp - may be off from actual temp (Pa and density too?)
                if ((SpawnPoints.Length > 0) && (selectedBody.isHomeWorld))
                {
                    Transform COL = SpawnPoints[0].GetSpawnPointTransform();
                    alt = selectedBody.GetAltitude(COL.position);
                }
                P = selectedBody.GetPressure(alt) * 1000;
                T = selectedBody.GetTemperature(alt);
            }
        }

        public void SetMaintainPressure(bool on)
        {
            MaintainPressure = on;
            for (int i = 0; i < envelopeList.Count; i++)
                envelopeList[i].SetVolumeHold(MaintainPressure);

            if (MaintainPressure)
            {
                Events["toggleMaintainPressure"].guiName = "Pressure Hold " + envelopeList[0].gasCells[0].targetPressure + " Pa";
                //double neededAir;
                PartResource prLift, prAir;
                if (HighLogic.LoadedScene == GameScenes.EDITOR)
                {
                    double P = 0;
                    double T = 0;
                    GetSpawnPointPT(ref P, ref T);
                    for (int i = 0; i < envelopeList.Count; i++)
                    {
                        prLift = envelopeList[i].part.Resources.Get(ModuleGasCell.resourceInfo[ModuleGasCell.HYDROGEN].id);
                        prAir = envelopeList[i].part.Resources.Get(ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].id);
                        if ((prLift != null) && (prAir != null))
                        {
                            double totalMolesNeeded = ModuleGasCell.CalcMoles(P, prLift.maxAmount, T);
                            double molesNeeded = totalMolesNeeded - envelopeList[i].gasCells[ModuleAerostat.LIFTCELL].gasMoles;
                            double molesMax = ModuleGasCell.CalcMoles(P, prAir.maxAmount, T);
                            if (molesNeeded <= molesMax)
                                prAir.amount = 0.5 * ModuleGasCell.calcVolume_L(molesNeeded, T, P);
                            else
                                prAir.amount = 0.5 * ModuleGasCell.calcVolume_L(molesMax, T, P);
                        }
                    }
                }
            }
            else
            {
                Events["toggleMaintainPressure"].guiName = "Pressure Hold Off";
            }
        }

        protected void SetAtmDensity()
        {
            ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].KgPerLiter = selectedBody.atmDensityASL * ModuleGasCell.METERS3_PER_LITER;
            ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].KgPerMole = selectedBody.atmDensityASL * ModuleGasCell.METERS3_PER_LITER * ModuleGasCell.LITER_PER_MOLE;
        }

        protected void GetPlanetsWithAtmosphereList()
        {
            bodiesWithAtmosphere.Clear();
            for (int i = 0; i < FlightGlobals.Bodies.Count; i++)
            {
                if ((FlightGlobals.Bodies[i].atmosphere) && (FlightGlobals.Bodies[i].name != "Sun"))
                {
                    bodiesWithAtmosphere.Add(FlightGlobals.Bodies[i]);
                    if ((FlightGlobals.Bodies[i].name == "Kerbin") &&
                        (currentBodyIndex == -1))
                        currentBodyIndex = bodiesWithAtmosphere.Count - 1;
                }
            }

            if (bodiesWithAtmosphere.Count > 0)
            {
                if (currentBodyIndex == -1)
                    currentBodyIndex = 0;
                selectedBody = bodiesWithAtmosphere[currentBodyIndex];

                if (currentBodyIndex >= 0)
                    Events["toggleBody"].guiName = "Body(" + selectedBody.name + ")";
                else
                    Events["toggleBody"].guiName = "Body()";
            }
        }

        protected void SetForAltitude(double setAltitude)
        {
            //Debug.Log(modTag + "-> SetForAltitude(" + setAltitude.ToString("F0") + ")");

            // GetDensity(P,T) is different from the part.atmDensity - get the ratio to scale the pressure at the desired alt.
            double pressureAtAlt = vessel.mainBody.GetPressure(vessel.altitude) * 1000.0 ; // KPa to Pa
            double temperatureAtAlt = vessel.mainBody.GetTemperature(vessel.altitude);
            double densityAtCurrentAlt = vessel.mainBody.GetDensity(pressureAtAlt, temperatureAtAlt) * ModuleGasCell.METERS3_PER_LITER; // Convert to Kg/liter
            double scale = part.atmDensity / densityAtCurrentAlt;

            pressureAtAlt = vessel.mainBody.GetPressure(setAltitude) * 1000.0; // KPa to Pa
            temperatureAtAlt = vessel.mainBody.GetTemperature(setAltitude);
            double densityAtAlt = vessel.mainBody.GetDensity(pressureAtAlt, temperatureAtAlt) * ModuleGasCell.METERS3_PER_LITER; // Convert to Kg/liter
            densityAtAlt = densityAtAlt * scale;
            double deltaDensity = (densityAtAlt - ModuleGasCell.resourceInfo[ModuleGasCell.HYDROGEN].KgPerLiter);
            double targetVolume = 1000 * vessel.GetTotalMass() * ModuleGasCell.METRIC_TONS_TO_KG / (deltaDensity * ModuleGasCell.liftMultiplier); // Convert Kt to Kg

            double totalLiftVolume = 0.0;
            for (int i = 0; i < envelopeList.Count; i++)
                if (envelopeList[i].gasCells[ModuleAerostat.LIFTCELL].gasLeak < ModuleGasCell.NOT_A_LEAK)
                {
                    totalLiftVolume += envelopeList[i].gasCells[ModuleAerostat.LIFTCELL].cellVolume;
                }

            double percentFull = targetVolume / totalLiftVolume;
            d1 = (float)totalLiftVolume;
            d2 = (float)targetVolume;
            d3 = (float)(100 * percentFull);
            for (int i = 0; i < envelopeList.Count; i++)
            {
                ModuleGasCell liftCell = envelopeList[i].gasCells[ModuleAerostat.LIFTCELL];
                if (liftCell.gasLeak < ModuleGasCell.NOT_A_LEAK)
                {
                    liftCell.targetMoles = (float)((1000 * pressureAtAlt * liftCell.cellVolume * percentFull) / (ModuleGasCell.R * temperatureAtAlt)); // N = PV / RT
                }
                else
                {
                    if (percentFull <= 1)
                        liftCell.targetMoles = 0.0f;
                }
            }

            //Debug.Log(modTag + "   SetForAltitude ->");
        }

        private int locateEnvelopes()
        {
            //Debug.Log(modTag + "-> locateEnvelopes");

            cellList.Clear();
            envelopeList.Clear();

            if (HighLogic.LoadedScene == GameScenes.FLIGHT)
            {
                if (vessel != null)
                {
                    cellList = vessel.FindPartModulesImplementing<ModuleGasCell>();
                    envelopeList = vessel.FindPartModulesImplementing<ModuleAerostat>();
                }
            }
            else if (HighLogic.LoadedScene == GameScenes.EDITOR)
            {
                if (EditorLogic.RootPart != null)
                {
                    cellList.AddRange(EditorLogic.FindObjectsOfType<ModuleGasCell>());
                    envelopeList.AddRange(EditorLogic.FindObjectsOfType<ModuleAerostat>());
                }
            }
                
            //GUIWindow.ChangeCellCount(cellList.Count);
            //Debug.Log("[AirshipControler] locateEnvelopes ->");
            return cellList.Count;
        }

        void OnGameSceneChange(GameScenes scene)
        {
            locateEnvelopes();
        }

        void OnEditorRestart()
        {
            locateEnvelopes();
        }

        void OnEditorScreenChange(EditorScreen screen)
        {
            locateEnvelopes();
        }

        void OnEditorPartEvent(ConstructionEventType evt, Part part)
        {
            //MonoBehaviour.print (evt.ToString ());
            switch (evt)
            {
                case ConstructionEventType.PartDeleted:
                case ConstructionEventType.PartAttached:
                case ConstructionEventType.PartDetached:
                    locateEnvelopes();
                    break;
            }
        }

        public void VesselModifiedEvent(Vessel v)
        // Watch for added/removed envelopes
        {
            //Debug.Log(modTag + "-> VesselModifiedEvent()");
            if (v == part.vessel)
                locateEnvelopes();
            //Debug.Log(modTag + "   VesselModifiedEvent ->");
        }

        protected void HoldPressure()
        {
            for (int i = 0; i < envelopeList.Count; i++)
                envelopeList[i].HoldPressure();
        }

        protected void HoldAltitude()
        {

        }

        protected void HoldPitch()
        {

        }

        protected void CalcEditorDisplayParams()
        {
            //Debug.Log(modTag + "-> CalcEditorDisplayParams()");

            if (EditorLogic.RootPart != null)
            {
                double AtmDensity;
                double alt = 0.0;

                // Get conditions at the spawn point (P, T)
                PSystemSetup.SpaceCenterFacility.SpawnPoint[] SpawnPoints = PSystemSetup.Instance.GetSpaceCenterFacility(EditorLogic.fetch.launchSiteName).spawnPoints;

                // Note - this retrieves the reference temp - may be off from actual temp (Pa and density too?)
                if ((SpawnPoints.Length > 0) && (selectedBody.isHomeWorld))
                {
                    Transform COL = SpawnPoints[0].GetSpawnPointTransform();
                    alt = selectedBody.GetAltitude(COL.position);
                }
                double P = selectedBody.GetPressure(alt) * 1000;
                double T = selectedBody.GetTemperature(alt);

                // Get total volumes
                totalVolume = 0.0f;
                double H2LiftMoles = 0.0;
                double balloonetVolume = 0.0;
                for (int i = 0; i < cellList.Count; i++)
                    if (cellList[i].gasIndex == ModuleGasCell.ATMOSPHERE)
                        balloonetVolume += cellList[i].cellVolume;
                    else
                    {
                        totalVolume += cellList[i].cellVolume;
                        H2LiftMoles += cellList[i].gasMoles;
                    }

                // Get lift gas current/max
                double H2CurLiters = 0.0;
                double H2MaxLiters = 0.0;
                List<PartResource> rl;
                foreach (Part p in EditorLogic.SortedShipList)
                {
                    rl = p.Resources.GetAll(ModuleGasCell.resourceInfo[ModuleGasCell.HYDROGEN].id);
                    for (int i = 0; i < rl.Count; i++)
                    {
                        H2CurLiters += rl[i].amount;
                        H2MaxLiters += rl[i].maxAmount;
                    }
                }
                curH2 = (float)(H2CurLiters + H2LiftMoles * ModuleGasCell.MOLES_PER_LITER);
                double totalCurMoles = H2LiftMoles + H2CurLiters * ModuleGasCell.LITER_PER_MOLE;
                double totalMaxMoles = (totalVolume + H2MaxLiters) * ModuleGasCell.LITER_PER_MOLE;

                AtmDensity = selectedBody.GetDensity(P * 0.001, T) * ModuleGasCell.KGMETERS3_TO_KGLITER * ModuleGasCell.liftMultiplier;

                // EditorLogic.fetch.ship.GetTotalMass() fails in editor if tweakscale scales a part - so we do it the hard way
                totalMass = 0.0f;
                foreach (Part p in EditorLogic.fetch.ship.parts)
                {
                    // p.GetResourceMass() is not working for envelopes ???
                    totalMass += p.mass + p.GetResourceMass();
                }
                totalMass = (float)(totalMass * ModuleGasCell.METRIC_TONS_TO_KG);

                // Add envelope gas mass as p.GetResourceMass() doesn't seem to capture it
                for (int i = 0; i < cellList.Count; i++)
                {
                    totalMass += (float)(cellList[i].gasMoles * ModuleGasCell.resourceInfo[cellList[i].gasIndex].KgPerMole);
                }

                maxLift = (float)(totalVolume * AtmDensity - totalMass);
                double CurVol = ModuleGasCell.calcVolume_L(H2LiftMoles, T, P);
                if (CurVol > totalVolume)
                    CurLift = (float)(totalVolume * AtmDensity);
                else
                    CurLift = (float)(ModuleGasCell.calcVolume_L(H2LiftMoles, T, P) * AtmDensity);
                totalBuoyancy = (float)(CurLift - totalMass);

                // Min H2 - only for non/semi rigid
                minH2 = (float)(ModuleGasCell.CalcMoles(P + 500, totalVolume - balloonetVolume, T) * ModuleGasCell.LITER_PER_MOLE);
                maxH2 = (float)(ModuleGasCell.CalcMoles(P + 500, totalVolume, T) * ModuleGasCell.LITER_PER_MOLE);

                // Max altitude estimate
                maxAlt = -100.0f;
                bool done = false;
                do
                {
                    P = selectedBody.GetPressure(maxAlt);
                    T = selectedBody.GetTemperature(maxAlt);
                    AtmDensity = selectedBody.GetDensity(P, T) * ModuleGasCell.KGMETERS3_TO_KGLITER * ModuleGasCell.liftMultiplier;
                    done = (totalVolume * AtmDensity) < totalMass;
                    if (!done)
                        maxAlt += 100;
                } while (!done);

            }

            //Debug.Log(modTag + "   CalcEditorDisplayParams ->");
        }

        protected void CalcFlightDisplayParams()
        {
            //Debug.Log(modTag + "-> CalcFlightDisplayParams()");

            CurLift = 0.0f;
            float curVolume = 0.0f;

            rate = (float)part.vessel.verticalSpeed;
            trim = (float)(Vector3d.Dot(FlightGlobals.ship_upAxis, vessel.GetTransform().up) * RadToDeg);
            TempDiff = (float)(part.temperature - vessel.atmosphericTemperature);

            maxPressureDiff = 0.0f;
            minPressureDiff = 10000000.0f;
            for (int i = 0; i < cellList.Count; i++)
            {
                curVolume += cellList[i].gasVolume;
                if (cellList[i].pressureDifferential > maxPressureDiff)
                    maxPressureDiff = cellList[i].pressureDifferential;
                if (cellList[i].pressureDifferential < minPressureDiff)
                    minPressureDiff = cellList[i].pressureDifferential;
            }

            CurLift = (float)(curVolume * part.atmDensity * ModuleGasCell.KGMETERS3_TO_KGLITER * ModuleGasCell.liftMultiplier);

            totalMass = (float)(vessel.GetTotalMass() * ModuleGasCell.METRIC_TONS_TO_KG);
            // vessel.GetTotalMass() appears to be missing the resource mass in the envelopes
            for (int i = 0; i < cellList.Count; i++)
                totalMass += (float)(cellList[i].gasMoles * ModuleGasCell.resourceInfo[cellList[i].gasIndex].KgPerMole);
            totalBuoyancy = (float)(CurLift - totalMass);

            //Debug.Log(modTag + "   CalcFlightDisplayParams ->");
        }

        protected void Test()
        {
            GetPlanetsWithAtmosphereList();
            Vector3 COL = part.rigidbody.worldCenterOfMass;
            double alt = selectedBody.GetAltitude(COL);
            double P = selectedBody.GetPressure(alt);
            double T = selectedBody.GetTemperature(alt);
            double AtmDensity = selectedBody.GetDensity(P, T);
            double atmDensity = part.atmDensity;
            Debug.Log(modTag + " a:" + alt.ToString("F0") + "  e:" + AtmDensity.ToString("F3") + "  f:" + atmDensity.ToString("F3") + "   r:" + atmDensity / AtmDensity);
        }

        private bool TimeForUpdate()
        {
            bool rtn = false;
            if (Time.fixedTime > NextUpdate)
            {
                NextUpdate = Time.fixedTime + 0.25f;
                rtn = true;
            }
            return rtn;
        }

        //------------------------------------------------------------------------------------------------//

        public override void OnStart(StartState state)
        {
            //Debug.Log(modTag + "-> OnStart()");
            base.OnStart(state);

            if (HighLogic.LoadedScene == GameScenes.EDITOR)
            {
                GetPlanetsWithAtmosphereList();
                GameEvents.onGameSceneLoadRequested.Add(OnGameSceneChange);
                GameEvents.onEditorPartEvent.Add(OnEditorPartEvent);
                GameEvents.onEditorRestart.Add(OnEditorRestart);
                GameEvents.onEditorScreenChange.Add(OnEditorScreenChange);
                SetAtmDensity();

                //Fields["totalBuoyancy"].guiName = "Max Buoyancy";
            }
            else
            {
                GameEvents.onVesselWasModified.Add(VesselModifiedEvent);

                LocalAtm = part.Resources.Get(ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].id);
                // Add a default atm intake if not already present
                if (LocalAtm == null)
                {
                    Debug.Log(modTag + "LocalAtm not found - creating..");
                    ConfigNode newConfigNode = new ConfigNode("RESOURCE");
                    newConfigNode.AddValue("name", ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].name);
                    newConfigNode.AddValue("maxAmount", "400");
                    newConfigNode.AddValue("amount", 0.0f);
                    newConfigNode.AddValue("flowState", "true");
                    LocalAtm = part.AddResource(newConfigNode);
                    if (LocalAtm == null)
                        Debug.Log(modTag + "Ambient Atm create failed");

                    //Fields["totalBuoyancy"].guiName = "Buoyancy";
                }
            }

            locateEnvelopes();

            GUIWindow = new AerostatControlGUI();

            //Debug.Log(modTag + "   OnStart ->");
        }

        public void OnDestroy()
        {
            //Debug.Log(modTag + "-> OnDestroy()");

            if (HighLogic.LoadedScene == GameScenes.EDITOR)
            {
                GameEvents.onGameSceneLoadRequested.Remove(OnGameSceneChange);
                GameEvents.onEditorPartEvent.Remove(OnEditorPartEvent);
                GameEvents.onEditorRestart.Remove(OnEditorRestart);
                GameEvents.onEditorScreenChange.Remove(OnEditorScreenChange);
            }
            else
                GameEvents.onVesselWasModified.Remove(VesselModifiedEvent);
            
            GUIWindow = null;
            //Debug.Log(modTag + "   OnDestroy ->");
        }

        public override void OnUpdate()
        {
            //Debug.Log(modTag + "-> OnUpdate");

            base.OnUpdate();
            CalcFlightDisplayParams();
            lastAltitude = (float)vessel.altitude;
            lastRate = rate;

            //GUIWindow.SetParams(envelopeList);

            //Debug.Log(modTag + "   OnUpdate ->");
        }

        public void FixedUpdate()
        {
            //Debug.Log(modTag + "-> FixedUpdate");

            if (HighLogic.LoadedScene == GameScenes.FLIGHT)
            {
                //Test();
                if (TimeForUpdate())  // Updates every half second
                {
                    if (MaintainPressure)  // This should be the main form of control
                        HoldPressure();
                }

                // Refill intake air - div by 2 so room for exaust as well as intake
                if (LocalAtm != null)
                    LocalAtm.amount = LocalAtm.maxAmount / 2;
                else
                    LocalAtm = part.Resources.Get(ModuleGasCell.resourceInfo[ModuleGasCell.ATMOSPHERE].id);
            }
            else if (HighLogic.LoadedScene == GameScenes.EDITOR)
            {
                CalcEditorDisplayParams();
            }

            //Debug.Log(modTag + "   FixedUpdate ->");
        }

    }

}
