using System.Collections.Generic;
using UnityEngine;

namespace DraggableMenu
{
    public class PositionController : MonoBehaviour
    {
        private UIPartActionWindow window;
        private EZScreenPlacement window_pos;

        private int child_count;
        private List<UIPartActionItem> items;
        private bool interactive;

        private bool frozen;
        private Vector3 frozen_pos;
        private Vector3 mousePosition;
        private float width, height, localOffsetX, localOffsetY;
        private bool isMousePressed, isAltPressed, wasLifted, draggingMode, isMoving, wasMoved;
        
        protected void Awake()
        {
            window = gameObject.GetComponent<UIPartActionWindow>();
            window_pos = gameObject.GetComponent<EZScreenPlacement>();
        }

        private void EnumerateItems()
        {
            child_count = transform.childCount;
            items = new List<UIPartActionItem>();
            
            for (int i = 0; i < child_count; i++)
            {
                var child = transform.GetChild(i).GetComponent<UIPartActionItem>();
                if (child != null)
                    items.Add(child);
            }
        }

        private bool IsInteractive(List<UIPartActionItem> items)
        {
            foreach (var item in items)
            {
                if (item is UIPartActionLabel || item is UIPartActionProgressBar)
                    continue;

                return true;
            }

            return false;
        }

        public void LateUpdate()
        {
            // Check if the window is valid or changed size.
            if (!window.isValid)
            {
                frozen = false;
                draggingMode = false;
                items = null;
                return;
            }

            if (items == null || transform.childCount != child_count)
            {
                EnumerateItems();

                interactive = IsInteractive(items);
            }

            isMousePressed = Input.GetMouseButton(0);
            isAltPressed = GameSettings.MODIFIER_KEY.GetKey();

            AutoLift();

            if (!draggingMode)
            {
                MenuStabilizer();
                return;
            }
            MenuMover();
        }

        // Lift window if it is off-screen.
        // (window_pos.screenPos.y > 0.5) check is meant to skip first window init.
        // Otherwise window would flicker at bottom-left corner.
        private void AutoLift()
        {
            if (!draggingMode && window_pos.screenPos.y > 0.5 && window_pos.screenPos.y < height)
            {
                window_pos.screenPos.y = height;
                window_pos.UpdateCamera();
                wasLifted = true;
            }
            else wasLifted = false;
        }

        // This checks if mouse is over menu window.
        // In case of MenuStabilizer by comparing mouse position with initial menu window position.
        // In case of Dragging Mode by comparing mouse position with actual menu coordinates.
        private bool IsMouseOver()
        {
            BoxCollider title_box = window.titleBar.collider as BoxCollider;
            Vector3 row_start = window.rowRootTransform.localPosition;

            mousePosition = Input.mousePosition;
            width = (title_box != null) ? title_box.size.x : row_start.x + 200;
            height = -row_start.y + window.rowHeight * items.Count;

            if (draggingMode)
                return (frozen_pos.x < mousePosition.x && mousePosition.x < frozen_pos.x + width &&
                        frozen_pos.y > mousePosition.y && mousePosition.y > frozen_pos.y - height);
            return new Rect(window_pos.screenPos.x, wasLifted ? 0 : window_pos.screenPos.y - height, width, height).Contains(mousePosition);
        }

        // This is instantiated for every part menu window and locks it position when moused over.
        private void MenuStabilizer()
        {
            // Unfreeze all in map view or when manipulating the camera.
            if (!interactive || MapView.MapIsEnabled ||
                (Input.GetMouseButton(1) && !isAltPressed) ||
                (Input.GetMouseButton(2) && !isAltPressed) ||
                GameSettings.AXIS_MOUSEWHEEL.GetAxis() != 0f ||
                GameSettings.ZOOM_IN.GetKey() || GameSettings.ZOOM_OUT.GetKey() ||
                GameSettings.CAMERA_ORBIT_LEFT.GetKey() || GameSettings.CAMERA_ORBIT_RIGHT.GetKey() ||
                GameSettings.CAMERA_ORBIT_UP.GetKey() || GameSettings.CAMERA_ORBIT_DOWN.GetKey())
            {
                frozen = false;
                return;
            }

            // Restore last position if frozen.
            if (frozen) UpdateMenuPosition();

            // Save position if moused is over menu.
            frozen = IsMouseOver();
            if (frozen) frozen_pos = window_pos.screenPos;

            // Switching to dragging mode.
            if (frozen && isMousePressed && isAltPressed) draggingMode = true;
        }

        // This moves menu window to current mouse position as long as ALT-Key and LMB are pressed.
        // When they are no longer pressed it restores last saved position until menu is closed or moved again.
        private void MenuMover()
        {
            // Unfreeze all in map view.
            if (!interactive || MapView.MapIsEnabled)
            {
                frozen = draggingMode = isMoving = wasMoved = false;
                return;
            }

            mousePosition = Input.mousePosition;

            // Calculating local offset.
            if (!isMoving && isMousePressed && isAltPressed && IsMouseOver())
            {
                localOffsetX = wasMoved
                    ? mousePosition.x - frozen_pos.x
                    : mousePosition.x - window_pos.screenPos.x;
                localOffsetY = wasMoved
                    ? frozen_pos.y - mousePosition.y
                    : window_pos.screenPos.y - mousePosition.y;

                // Adjust vertical offset in case if menu was autolifted.
                if (!wasMoved && mousePosition.y > window_pos.screenPos.y) localOffsetY = height - mousePosition.y;
                
                isMoving = wasMoved = true;
            }

            if (isMoving && isMousePressed && isAltPressed)
            {
                // Applying changes.
                frozen_pos.x = mousePosition.x - localOffsetX;
                frozen_pos.y = mousePosition.y + localOffsetY;
            }
            else isMoving = false;

            UpdateMenuPosition();
        }

        // Updating menu coordinates.
        private void UpdateMenuPosition()
        {
            frozen_pos.z = window.zOffset;
            window_pos.screenPos = frozen_pos;
            window_pos.UpdateCamera();
        }
    }
}
