﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using UnityEngine;

namespace TrekDrive
{
	//###################################################################################################
	// Warp Coil Module
	// Attached to Nacelle parts; Nacelles must be charged for the drive to work.
	// Requires Warp Plasma
	//###################################################################################################
	public class SW_ModuleWarpCoil : PartModule
	{
		[KSPField]
		public float minWarpPlasma = 100f;

		[KSPField]
		public float warpPlasmaConsumed = 5f;

		[KSPField]
		public float warpPlasmaNeeded = 250f;

		[KSPField]
		public double cutoffThreshold = 2d;

		// Used in temperature increase calculations
		// Given in decimal form of a percentage.
		[KSPField]
		public float coilEfficiency = 0.5f;

		[KSPField]
		public double maxWarp = 1d;

		// Variables needed to charge the coils
		private double warpPlasma = 0d;
		private int warpPlasmaID = PartResourceLibrary.Instance.GetDefinition("WarpPlasma").id;
		private double warpPlasmaAmnt;
		private double warpPlasmaMaxAmnt;
		protected bool chargeWarpCoils = false;

		// Variable needed to produce Liquid Deuterium
		private int deutID = PartResourceLibrary.Instance.GetDefinition("LqdDeuterium").id;

		[KSPField(isPersistant = true)]
		public bool coilCharged = false;

		// Initiates charging of the warp coil
		[KSPEvent(groupName = "CoilPAW", guiActive = true, active = true, guiActiveEditor = false, guiName = "Charge Warp Coils", guiActiveUnfocused = false)]
		public void ChargeCoil()
		{
			chargeWarpCoils = true;
			//chargeStatus = "Charging";
		}

		// KSPAction to allow the warp coil charge to be triggered with an action group
		[KSPAction(guiName = "Charge Warp Coils", isPersistent = true)]
		public void ChargeCoilAction(KSPActionParam param)
		{
			ChargeCoil ();
		}

		// Added 13 July 2021:  Variable and PAW code necessary to allow discharging of the warp coils.
		// This will allow users to simulate a full shutdown, at least visually
		protected bool dischargeCoils = false;

		// Initiates discharging of the warp coil
		[KSPEvent(groupName = "CoilPAW", guiActive = true, active = true, guiActiveEditor = false, guiName = "Discharge Warp Coils", guiActiveUnfocused = false)]
		public void DischargeCoil()
		{
			dischargeCoils = true;
			//chargeStatus = "Charging";
		}

		// KSPAction to allow the warp coil discharge to be triggered with an action group
		[KSPAction(guiName = "Discharge Warp Coils", isPersistent = true)]
		public void DischargeCoilAction(KSPActionParam param)
		{
			DischargeCoil ();
		}

		[KSPField(groupName = "CoilPAW", groupDisplayName = "Warp Coil Operation", guiName = "Charge Status", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public string chargeStatus = "Not Charging";

		[KSPField(groupName = "CoilPAW", guiName = "Charge Percentage", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public float percentCharged = 0f;

		// Animation variable declaration
		float newNormalizedTime;

		// Set the module to be stageable
		public override bool IsStageable()
		{
			return true;
		}

		// Called when the part is started by Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);
			this.part.stagingIcon = "LIQUID_ENGINE";
			this.part.stackIconGrouping = StackIconGrouping.SAME_MODULE;

			//Debug.Log ("[TrekDrive] is stageable " + this.stagingEnabled);
		}

		// Called on part loading, hopefully will fix the unresponsiveness on switching and loading from save
		public override void OnLoad(ConfigNode node)
		{
			if (HighLogic.LoadedSceneIsFlight)
			{
				if (percentCharged > 0 && chargeStatus == "Charged")
				{
					float colorValue = (float)(percentCharged / 100);
					var glowColor = new Color (colorValue, colorValue, colorValue);
					this.part.GetPartRenderers ().FirstOrDefault ().material.SetColor ("_EmissiveColor", glowColor);

					// This is necessary to set the charge state of the coil
					coilCharged = true;
				}
			}
		}

		// Variable definitions for the throttle ("throttle") and active vessel ("ship").
		// These definitions will make the code neater as it will cut down on long strings and make it easier to read
		private float throttle;
		private Vessel ship;

		// Fixed Update Function
		public void FixedUpdate()
		{
			// Set the "throttle" variable only if we are in flight, and get the connected resource totals for
			// electric charge only if we are in flight as well as the "ship" variable to hold the active vessel.
			if (HighLogic.LoadedSceneIsFlight)
			{
				// Set the "throttle" variable with the current mainThrottle percentage
				throttle = FlightGlobals.ActiveVessel.ctrlState.mainThrottle;

				// Set "ship" to the active vessel to simplify the overall code.
				ship = FlightGlobals.ActiveVessel;
			}


			// Check to make sure there is a warp core on the vessel
			if (chargeWarpCoils && HighLogic.LoadedSceneIsFlight)
			{
				bool hasWarpCore = false;
				for (int i = 0; i < ship.parts.Count; i++)
				{
					PartModuleList partModules = ship.parts [i].Modules;
					for (int j = 0; j < partModules.Count; j++)
					{
						if (partModules[j] is SW_ModuleWarpCore)
						{
							hasWarpCore = true;
						}
					}
				}

				// If there is a warp core present, allow the coils to charge
				// If not, set chargeWarpCoils to false, print warning to screen
				if (hasWarpCore)
				{
					chargeWarpCoils = true;
				}
				else
				{
					chargeWarpCoils = false;
					ScreenMessages.PostScreenMessage ("NO WARP CORE INSTALLED - CANNOT CHARGE COILS!");
				}
			}

			// Charge the warp coils if we have clicked the toggle, the coils aren't charged, or if charge percentage is less than 100
			// Update - Feb 2021:  Added to the conditions to allow the coils to be recharged when they have been partially discharged
			// This was not possible before, as the conditional was set up to only account for a completely discharged/uncharged coil
			// I also added an ability to stop/prevent charging and post a message to the screen if there is not enough, or any
			// warp plasma with which to charge the coils.
			if ((chargeWarpCoils && !coilCharged && percentCharged < 100d && HighLogic.LoadedSceneIsFlight) || (chargeWarpCoils && coilCharged && percentCharged < 100d && HighLogic.LoadedSceneIsFlight))
			{
				// Set the charge status
				if (chargeWarpCoils)
				{
					chargeStatus = "Charging...";
				}

				// Get the amount of WarpPlasma in the vessel, and the total possible amount of WarpPlasma
				//this.vessel.GetConnectedResourceTotals(warpPlasmaID, out warpPlasmaAmnt, out warpPlasmaMaxAmnt);
				this.vessel.GetConnectedResourceTotals(warpPlasmaID, out warpPlasmaAmnt, out warpPlasmaMaxAmnt, true);

				// If there is warpPlasma transfer warpPlasma to the coil
				if (warpPlasmaAmnt > 0)
				{
					// Amount of Warp Plasma to Transfer
					float wpTransferAmnt = warpPlasmaConsumed * TimeWarp.fixedDeltaTime;

					// First line depletes the warpPlasma in the vessel
					// Second line fills the coil with that warpPlasma
					this.vessel.RequestResource (this.part, warpPlasmaID, wpTransferAmnt, true);
					this.part.RequestResource(warpPlasmaID, -wpTransferAmnt, ResourceFlowMode.ALL_VESSEL, true);

					// Update the percentage of charge
					warpPlasma += wpTransferAmnt;
					//this.part.GetConnectedResourceTotals(warpPlasmaID, out warpPlasma, out warpPlasmaMaxAmnt, true);
					percentCharged = (float)warpPlasma / (float)warpPlasmaNeeded * 100f;

					// Update the _EmissiveColor of the first/default renderer on the nacelle part.
					// The first/default renderer is the first mesh in the heirarchy.  Thus, modders must ensure
					// that the bussard and field emitter (red and blue portions respectively) be part of the first mesh in the
					// Unity/Mesh heirarchy.
					newNormalizedTime = (float)(percentCharged / 100d);

					var glowColor = new Color (newNormalizedTime, newNormalizedTime, newNormalizedTime);
					this.part.GetPartRenderers ().FirstOrDefault ().material.SetColor ("_EmissiveColor", glowColor);

				}
				else
				{
					// If we try to charge the coil, but there is no warp plasma with which to charge the coils
					// post a message to the screen and set the toggle to false.
					// The chargeStatus is not changed.  This means that charging stops and if the coils is already labeled as
					// "Charged" the coil can still be used to travel at warp until it is completely discharged.
					ScreenMessages.PostScreenMessage ("NOT ENOUGH WARP PLASMA TO CHARGE COILS!");
					chargeWarpCoils = false;
				}

				// If the coil is charged, stop charging
				if (warpPlasma >= warpPlasmaNeeded)
				{
					coilCharged = true;
					chargeWarpCoils = false;
					chargeStatus = "Charged";
					Debug.Log ("[TrekDrive] Coil Charged.");
				}
			}
			else
			{
				chargeWarpCoils = false;
			}

			//########################
			// Added 13 July 2021:  Here is the code to manually discharge the warp coils
			//########################
			if (HighLogic.LoadedSceneIsFlight && percentCharged > 0d && dischargeCoils)
			{
				chargeStatus = "Discharging...";

				float percentDecrease = (100*warpPlasmaConsumed * TimeWarp.fixedDeltaTime) / warpPlasmaNeeded;

				percentCharged -= percentDecrease;

				newNormalizedTime = (float)(percentCharged / 100d);

				var glowColor = new Color (newNormalizedTime, newNormalizedTime, newNormalizedTime);
				this.part.GetPartRenderers ().FirstOrDefault ().material.SetColor ("_EmissiveColor", glowColor);

				// If the coil is completely discharged, stop the discharge
				if (percentCharged <= 0d)
				{
					warpPlasma = 0;
					dischargeCoils = false;
					coilCharged = false;
					chargeStatus = "Not Charged";
					Debug.Log ("[TrekDrive] Coil Discharged.");
				}
			}

			// Variables to drain the warp core before discharging the nacelles
			//double drainCore = 0d;
			//bool warpCoreDrained = false;

			// Variable to hold the maximum warp factor
			double maxDriveWarp = 0d;
			//int totalNacelles = 0;

			// Variable to hold the ratio of the active drive's maximum warp rating to the nacelle's maximum warp rating
			double warpRatio = 1d;

			// Check to see if the warp drive is active
			bool isDriveActive = false;
			if (HighLogic.LoadedSceneIsFlight && !isDriveActive)	// Added "&& !isDriveActive" so that the check is not done when the drive is active
			{
				for (int i = 0; i < ship.parts.Count; i++)
				{
					PartModuleList partModules = ship.parts [i].Modules;
					for (int j = 0; j < partModules.Count; j++)
					{
						if (partModules[j] is SW_ModuleWarpGenerator)
						{
							isDriveActive = ship.parts [i].FindModulesImplementing<SW_ModuleWarpGenerator> ().First ().driveActive;
							maxDriveWarp = ship.parts [i].FindModulesImplementing<SW_ModuleWarpGenerator> ().First ().maxWarp;
							//totalNacelles = FlightGlobals.ActiveVessel.parts [i].FindModulesImplementing<SW_ModuleWarpGenerator> ().First ().numNacelles;
						}

						if (isDriveActive)
						{
							break;
						}
					}

					if (isDriveActive)
					{
						warpRatio = maxDriveWarp / maxWarp;
						break;
					}
				}
			}

			// Drain the warp core of Warp Plasma while at warp
			// Discharge the nacelles once the core is drained
			// Update Jan 2022:  Added "&& coilCharged" to this check to ensure that non-charghed extra coils don't cause a ship
			//                   to not go to warp if the minimum number of nacelles have been charged.
			if (HighLogic.LoadedSceneIsFlight && throttle > 0f && isDriveActive && coilCharged)
			{
				// Get the amount of Warp Plasma in the vessel
				this.vessel.GetConnectedResourceTotals (warpPlasmaID, out warpPlasmaAmnt, out warpPlasmaMaxAmnt);

				// Consume the Warp Plasma using flow priority, which should
				// drain the warp core first, then the nacelles.
				this.vessel.RequestResource (this.part, warpPlasmaID, warpPlasmaConsumed * throttle * TimeWarp.fixedDeltaTime, true);

				// As part of the new power system, the nacelle will generate Liquid Deuterium as the warp plasma is consumed.
				// This emulates the behavior of the plasma neutralizing back to deuterium. It would be gaseous, but we'll go to liquid
				// for simplicity.  The amount of liquid deuterium produced will be equal to the amount of warp plasma consumed.
				//this.vessel.RequestResource (this.part, deutID, -warpPlasmaConsumed * throttle * TimeWarp.fixedDeltaTime, true);

				// Check that coilEfficiency is a value between 0 and 1
				// First take the absolute value to ensure sanity, then see if efficiency is greater than 1
				coilEfficiency = Math.Abs(coilEfficiency);
				if (coilEfficiency > 1f)
				{
					// If efficiency is greater than 1, randomly choose an efficiency.
					// This is to discourage blind cheating (although an efficiency of 1 is already cheating).
					coilEfficiency = UnityEngine.Random.Range (0f, 1f);
				}

				// Increase the temperature of the nacelle while at warp
				// It follows an exponential behavior scaled by the coil efficiency
				// If the coil is 100% efficient (coilEfficiency = 1.0f) the part temperature will not increase from use.
				// Update - Jan. 2021:  Added a "warpRatio" varible that scales the overall exponential equation such that
				// when the coil's max warp is less than the drive's max warp, the coil will heat up faster, emulating the drive pumping
				// more plasma than the coil can handle, and if it's the other way around, it won't heat up as quickly.
				this.part.temperature += warpRatio * 2 * Math.Exp(3*(maxDriveWarp*throttle - maxDriveWarp))*(1 - coilEfficiency);

				// Update the charge state of the coil
				double coilMaxWP;		// Total Warp Plasma that can be held in the coil
				double coilWP;			// Current amount of Warp Plasma in the coil
				//this.part.GetConnectedResourceTotals (warpPlasmaID, out coilWP, out coilMaxWP);
				this.part.GetConnectedResourceTotals (warpPlasmaID, ResourceFlowMode.NO_FLOW, out coilWP, out coilMaxWP, false);
				//percentCharged = coilWP / coilMaxWP * 100d;
				//percentCharged = (coilWP/totalNacelles)/warpPlasmaNeeded * 100d;
				if (coilWP < warpPlasmaConsumed*throttle*TimeWarp.fixedDeltaTime)
				{
					percentCharged -= warpPlasmaConsumed * throttle * TimeWarp.fixedDeltaTime;
				}
				//percentCharged = coilWP;

				// Call the function to update the nacelle glow animation
				//UpdateNacelleAnimation();

				// If the coild is completely discharged, set the variables to allow recharging
				// once more warp plasma is produced.
				if (percentCharged < cutoffThreshold)
				{
					ScreenMessages.PostScreenMessage ("NOT ENOUGH WARP PLASMA!");
					coilCharged = false;
					chargeStatus = "Not Charged";
					warpPlasma = 0d;
				}
			}
		}
	}






	//###################################################################################################
	// Warp Core Module
	// Generates Warp Plasma from matter and anti-matter,
	// Also generates electricity (subbing for EPS system)
	//###################################################################################################
	public class SW_ModuleWarpCore : PartModule
	{
		// The minimum amount of Anti-Matter necessary to produce warp plasma
		[KSPField]
		public float minAntiMatter = 0.05f;

		// The minimum amount of Matter necessary to produce warp plasma
		[KSPField]
		public float minMatter = 0.05f;

		// The amount of warp plasma produced per second
		[KSPField]
		public float warpPlasmaProdRate = 10f;

		// The amount of electric charge produced per second
		[KSPField]
		public float ecProdRate = 100f;

		// The path to the ambience sound effect
		[KSPField]
		public string ambientSoundPath = "TrekDrive/Sounds/tng_engine_hum";

		// Toggle button to activate the warp core to produce Warp Plasma and Electric Charge
		[KSPField(groupName = "CorePAW", groupDisplayName = "Warp Core Operation", guiName = "Warp Core Status", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Inactive", enabledText = "Active")]
		protected bool activateWarpCore = false;

		// KSPAction to allow the warp core to be toggle through an action group
		[KSPAction(guiName = "Toggle Warp Core",isPersistent = true)]
		public void ToggleWarpCore(KSPActionParam param)
		{
			if (!activateWarpCore)
			{
				activateWarpCore = true;
			}
			else
			{
				activateWarpCore = false;
			}
		}

		// Variable to control the sound
		private AudioSource engineHum;
		// Variable to control the sound playback
		private bool playSound = true;

		// Initialize the part when the part is started in Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);

			// Initialize the sound gameObject
			if (HighLogic.LoadedSceneIsFlight)
			{
				engineHum = this.part.gameObject.AddComponent<AudioSource>();
				engineHum.clip = GameDatabase.Instance.GetAudioClip (ambientSoundPath);
				engineHum.volume = GameSettings.SHIP_VOLUME;
				//engineHum.audio.volume = 1.0f;

				engineHum.panStereo = 0;
				engineHum.rolloffMode = AudioRolloffMode.Linear;
				engineHum.loop = true;
				engineHum.spatialBlend = 1f;
			}

		}

		// Variables needed to run the warp core
		private double antiMatAmnt;
		private double antiMatMaxAmnt;
		private double matterAmnt;
		private double matterMaxAmnt;
		private double warpPlasmaAmnt;
		private double warpPlasmaMaxAmnt;
		private double ecAmnt;
		private double ecMaxAmnt;
		public int antiMatID = PartResourceLibrary.Instance.GetDefinition("Antimatter").id;
		public int matterID = PartResourceLibrary.Instance.GetDefinition("LqdDeuterium").id;
		public int warpPlasID = PartResourceLibrary.Instance.GetDefinition("WarpPlasma").id;
		public int ecID = PartResourceLibrary.Instance.GetDefinition("ElectricCharge").id;

		// Called each frame, where the code to run the core resides
		public void FixedUpdate()
		{
			if (activateWarpCore)
			{
				this.vessel.GetConnectedResourceTotals(antiMatID, out antiMatAmnt, out antiMatMaxAmnt);
				this.vessel.GetConnectedResourceTotals (matterID, out matterAmnt, out matterMaxAmnt);
				this.vessel.GetConnectedResourceTotals(warpPlasID, out warpPlasmaAmnt, out warpPlasmaMaxAmnt);
				this.vessel.GetConnectedResourceTotals (ecID, out ecAmnt, out ecMaxAmnt);

				if (antiMatAmnt > minAntiMatter && matterAmnt > minMatter && (warpPlasmaAmnt < warpPlasmaMaxAmnt || ecAmnt < ecMaxAmnt))
				{
					// Consume AntiDeuterium and Deuterium
					//this.vessel.RequestResource (this.part, antiMatID, (minAntiMatter * 2f) * TimeWarp.fixedDeltaTime, true);
					//this.vessel.RequestResource (this.part, matterID, (minMatter * 2f) * TimeWarp.fixedDeltaTime, true);
					this.part.RequestResource(antiMatID, (minAntiMatter * 1d) * TimeWarp.fixedDeltaTime);
					this.part.RequestResource (matterID, (minMatter * 1d) * TimeWarp.fixedDeltaTime);

					// Produce WarpPlasma and ElectricCharge
					this.vessel.RequestResource(this.part, warpPlasID, -warpPlasmaProdRate * TimeWarp.fixedDeltaTime, false);
					this.vessel.RequestResource (this.part, ecID, -ecProdRate * TimeWarp.fixedDeltaTime, true);

				}
				/*else
				{
					// If we have tried to activate the warp core, but there aren't enough resources, or if the resources run
					// out while it is active, deactivate the core.
					activateWarpCore = false;
					engineHum.Stop ();
					playSound = true;
				}*/

				// Play ambient sound
				if (playSound)
				{
					engineHum.Play ();
					playSound = false;
				}
			}
			else
			{
				if (!playSound)
				{
					engineHum.Stop ();
					playSound = true;
				}
			}
		}
	}






	//###################################################################################################
	// Warp Field Generator
	// Controls the Warp Coils to create and modify the Warp Field;
	// This is what moves the ship.
	// Requires a minimum number of Nacelles, as well as a Warp Core to function
	//###################################################################################################
	public class SW_ModuleWarpGenerator : ModuleEngines			// Was PartModule, changed to allow the use of "engineID" for Waterfall warp effect
	{
		// Set the maxThrust of ModuleEngines to 0 to ensure no actual thrust is added to the ship
		[KSPField]
		public float maxThrust = 0f;

		// Maximum warp factor atainable, defaults to Warp 5
		[KSPField]
		public double maxWarp = 1d;

		// Minimum number of nacelles for the drive to operate
		// Defaults to 2, but I will enforce at least 1
		[KSPField]
		public int minNacelles = 1;

		// Amount of electric charge required per second while the drive is active.
		// The default value is 50 ec/s, and is consumed once the drive is in the "Active"
		// state.  The usage amount increases as the warp factor increases, doubling at maximum warp.
		[KSPField]
		public double electricityReq = 50d;

		// Get the resource ID for "ElectricCharge"
		public int ecID = PartResourceLibrary.Instance.GetDefinition("ElectricCharge").id;

		// A toggle to Set up the drive by checking for the coil charge state
		[KSPEvent(groupName = "GeneratorPAW", guiActive = true, active = true, guiActiveEditor = false, guiName = "Check Ready Status", guiActiveUnfocused = false)]
		public void CheckStatus()
		{
			driveStatus = "Checking...";
			numCoilsCharged = 0;
			CheckCharge ();
			Debug.Log("[TrekDrive] Check Status executed.");
		}

		// The KSPAction setup to allow the status check to be a setable action group
		[KSPAction(guiName = "Check Ready Status", isPersistent = true)]
		public void CheckStatusAction(KSPActionParam param)
		{
			CheckStatus ();
		}

		// Displays the status of the drive in the PAW
		[KSPField(groupName = "GeneratorPAW", groupDisplayName = "Warp Field Generator Operation", guiName = "Ready Status", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public string driveStatus = "Not Ready";

		// Toggle switch to activate the drive
		[KSPField(groupName = "GeneratorPAW", guiName = "Engage Warp Drive", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Drive Inactive", enabledText = "Drive Engaged")]
		public bool driveActive = false;

		// KSPAction to toggle to activate/deactivate the drive from an action group
		[KSPAction(guiName = "Toggle Warp Drive", isPersistent = true)]
		public void DriveActiveToggle(KSPActionParam param)
		{
			if (!driveActive)
			{
				driveActive = true;
				this.Activate ();
			}
			else
			{
				driveActive = false;
				this.Shutdown ();
			}
		}

		// Toggle switch to change orbit mode from Easy to Realistic, defaults to Easy
		[KSPField(groupName = "GeneratorPAW", guiName = "Orbital Mode", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Realistic", enabledText = "Easy")]
		public bool orbitMode = true;

		// KSPAction to toggle the orbit mode as an action group
		[KSPAction(guiName = "Toggle Orbit Mode", isPersistent = true)]
		public void ToggleOrbitMode(KSPActionParam param)
		{
			if (!orbitMode)
			{
				orbitMode = true;
			}
			else
			{
				orbitMode = false;
			}
		}

		// Slider to choose the current desired maximum warp factor
		[KSPField(groupName = "GeneratorPAW", isPersistant = true, guiActive = true, guiActiveEditor = true, guiName = "Warp Factor Selection"), UI_FloatRange(minValue = 0.25f, maxValue = 10f, stepIncrement = 0.25f)]
		public float selectedWarp = 1f;

		// Displays the current warp factor in the PAW
		[KSPField(groupName = "GeneratorPAW", guiName = "Current Warp Factor", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public float currentWF = 0f;

		// Variable telling us if there are enough nacelles to activate the drive
		protected bool enoughWarpCoils;

		// Function to enforce the requirement of at least 1 nacelle
		public int EnforceNacelleReq()
		{
			if (minNacelles < 1)
			{
				minNacelles = 1;
				Debug.Log("[TrekDrive] Number of Nacelles required to operate: "+minNacelles);
				return minNacelles;
			}
			else
			{
				Debug.Log("[TrekDrive] Number of Nacelles required to operate: "+minNacelles);
				return minNacelles;
			}
		}

		// Called when the part is started by Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);
			//EnforceNacelleReq();
			base.Events["Activate"].guiActive = false;
			base.Events ["Shutdown"].guiActive = false;

			((UI_FloatRange)Fields ["selectedWarp"].uiControlFlight).maxValue = (float)maxWarp;
			//this.selectedWarp = (float)maxWarp;
		}
			

		// Called when the part is made active
		public override void OnActive()
		{
			if (HighLogic.LoadedSceneIsFlight)
			{
				// Call the nacelle requirement enforcement function
				// just to ensure it gets set correctly.
				EnforceNacelleReq ();
				SetupDrive ();
			}
		}

		// Called when the part is loaded, hopefully will fix non-responsive problem
		public override void OnLoad(ConfigNode node)
		{
			if (HighLogic.LoadedSceneIsFlight)
			{
				EnforceNacelleReq ();
				SetupDrive ();
				//CheckStatus ();

				if (driveStatus == "Ready")
				{
					Debug.Log ("[TrekDrive] Onload variable setup.");
					allCoilsCharged = true;
					//driveStatus = "Not Ready";
					driveActive = false;
				}
			}
		}

		// Called when the part is unpacked.  This happens when the vessel is switched to from another vessel,
		// or when another vessel docks with it and becomes the active vessel
		protected void OnUnpack()
		{
			if (HighLogic.LoadedSceneIsFlight)
			{
				EnforceNacelleReq ();
				SetupDrive ();

				if (driveStatus == "Ready")
				{
					Debug.Log ("[TrekDrive] OnUnpack variable setup.");
					allCoilsCharged = true;
					//driveStatus = "Not Ready";
					driveActive = false;
				}
			}
		}


		// Variable to perform the drive setup
		public int numNacelles;

		// SetupDrive function:
		// Check to ensure that the minimum number of nacelles are present on the vessel
		// 17 July 2021 - Changed all "FlightGlobals.ActiveVessel" instances in this function
		// to "this.vessel".  I believe this is causing the issue with docking to the warp powered vessel.
		private void SetupDrive()
		{
			numNacelles = 0;
			Debug.Log ("[TrekDrive] Drive Setup Started.");

			for (int i = 0; i < this.vessel.parts.Count; i++)
			{
				PartModuleList partModules = this.vessel.parts [i].Modules;
				for (int j = 0; j < partModules.Count; j++)
				{
					if (partModules[j] is SW_ModuleWarpCoil && this.vessel.parts[i] != this.part)
					{
						numNacelles += 1;
						break;
					}
				}
			}

			// If the number of nacelles is equal to or greather than minNacelles
			// then set enoughWarpCoils to "true".
			if (numNacelles >= minNacelles)
			{
				enoughWarpCoils = true;
				Debug.Log ("[TrekDrive] Enough warp coils have been detected");
			}

			// Initialize the lastSelectedWarp varible
			lastSelectedWarp = selectedWarp;
		}

		// OtherActive function:
		// Checks to see if there are other active drives.  This is a sanity checker to ensure that only one
		// drive is active at a time.
		private bool OtherActive()
		{
			if (HighLogic.LoadedSceneIsEditor)
			{
				return false;
			}

			bool isDriveActive = false;
			for (int i = 0; i < FlightGlobals.ActiveVessel.parts.Count; i++)
			{
				PartModuleList partModules = FlightGlobals.ActiveVessel.parts [i].Modules;
				for (int j = 0; j < partModules.Count; j++)
				{
					if (partModules[j] is SW_ModuleWarpGenerator && FlightGlobals.ActiveVessel.parts[i] != this.part)
					{
						isDriveActive = FlightGlobals.ActiveVessel.parts [i].FindModulesImplementing<SW_ModuleWarpGenerator> ().First ().driveActive;
					}

					if (isDriveActive)
					{
						return true;
					}
				}
			}

			return false;
		}

		// Variables needed for additional coil checks
		private bool allCoilsCharged = false;
		private int numCoilsCharged = 0;

		// CheckCharge function:
		// Checks to make sure that all warp coils are charged before the drive can be engaged.
		private void CheckCharge()
		{
			for (int i = 0; i < FlightGlobals.ActiveVessel.parts.Count; i++)
			{
				PartModuleList partModules = FlightGlobals.ActiveVessel.parts [i].Modules;
				for (int j = 0; j < partModules.Count; j++)
				{
					if (partModules[j] is SW_ModuleWarpCoil && FlightGlobals.ActiveVessel.parts[i].FindModulesImplementing<SW_ModuleWarpCoil>().First().coilCharged)
					{
						numCoilsCharged += 1;
						//Debug.Log ("[TrekDrive] Number of Charged Coils: " + numCoilsCharged);
					}
				}
			}

			//Debug.Log ("[TrekDrive] numCoilsCharged: " + numCoilsCharged);
			//Debug.Log ("[TrekDrive] numNacelles: " + numNacelles);

			// If the number of charged coils is equal to the number of nacelles
			// then report that all coils are charged.
			// This means that although you may use more than the minimum number of nacelles
			// that they must all still be charged for the drive to function.
			//
			// Update Jan 2021: Added the OtherActive sanity check to make sure that there are no other active drives.
			// Update Jan 2022: Changed "numNacelles" to "minNacelles" to allow a mothership to go to warp if warp-capable shuttles are docked.
			if (numCoilsCharged == minNacelles && !OtherActive())
			{
				allCoilsCharged = true;
				driveStatus = "Ready";
				//driveActive = true;
			}
			else
			{
				allCoilsCharged = false;
				driveStatus = "Not Ready";
				//driveActive = false;
			}
		}


		// Initialize the drive, since apparently it's not getting set up when it should
		private bool initializeDrive = true;

		// Variables to control warp flight
		public float previousThrotState = 0f;
		public double orbitalSpeed;
		public double GravityBrakes;

		private float throttle;
		private Vessel ship;
		private double ecAmnt;
		private double ecMaxAmnt;
		private float lastSelectedWarp;

		// Variable to determine when timewarp has ended
		private bool prevTimeWarpState = false;
		private bool timeWarpEnded = false;

		// The Fixed Update function, the meat of the plugin
		public void FixedUpdate()
		{
			// Initialize the Drive
			if (HighLogic.LoadedSceneIsFlight && initializeDrive)
			{
				EnforceNacelleReq ();
				SetupDrive ();
				initializeDrive = false;
			}

			// variable to hold the current throttle value
			//float throttle = FlightInputHandler.state.mainThrottle;
			//float throttle;

			// Get the current amount of electric charge in the vessel, to be used in cutting off the drive
			// if we run out of electricity.  So, if the warp core can't keep up with the field generator,
			// the generator can't operate.
			//double ecAmnt;
			//double ecMaxAmnt;

			// Set the "throttle" variable only if we are in flight, and get the connected resource totals for
			// electric charge only if we are in flight as well.
			// Updated 13 July 2021: Added the "ship" variable to hold the active vessel.  This makes the code a little neater.
			if (HighLogic.LoadedSceneIsFlight)
			{
				// Set the "throttle" variable with the current mainThrottle percentage
				throttle = FlightGlobals.ActiveVessel.ctrlState.mainThrottle;

                // Set "ship" to the active vessel to simplify the overall code.
                ship = FlightGlobals.ActiveVessel;

				// Get the amount of electric charge in the vessel.
				this.vessel.GetConnectedResourceTotals (ecID, out ecAmnt, out ecMaxAmnt, true);

				// Activate the engine if the warp drive is active.
				if (driveActive)
				{
					this.Activate ();
				}
				else
				{
					this.Shutdown ();
				}

				// Call the UpdateThrottle function of ModuleEngines to see if it will allow Waterfall to function properly
				UpdateThrottle();
			}

			// This will prevent the drive from being activated if the coils are not charged.
			// If the coils aren't charged it will set the active status back to "false", but it
			// will allow activation if all coils are charged.
			if (HighLogic.LoadedSceneIsFlight && driveActive && !allCoilsCharged || OtherActive())
			{
				driveActive = false;
				this.Shutdown ();
			}

			// Set the value of the current warp factor (currentWF) for display in the PAW
			// Only if the drive is active
			if (HighLogic.LoadedSceneIsFlight && driveActive)
			{
				//currentWF = (float)(maxWarp) * (float)(throttle);
				currentWF = selectedWarp * throttle;
			}

			// Get the original orbital speed when the drive is active and the previous frame the trottle was at 0
			if (HighLogic.LoadedSceneIsFlight && driveActive && previousThrotState == 0f)
			{
				orbitalSpeed = Vector3d.Magnitude (ship.orbit.GetRelativeVel ());

				// Consume electric charge once the drive is active.  This emulates the drive maintaining a stable
				// warp field, even though the ship is not traveling at warp.
				//this.vessel.RequestResource (this.part, ecID, electricityReq * TimeWarp.fixedDeltaTime, true);
				this.part.RequestResource (ecID, electricityReq * TimeWarp.fixedDeltaTime);
			}

			// If there is not enough electric charge, shut down the dirve
			// NOTE:  THIS MAY BE THE CAUSE OF AN ORBIT ISSUE.  IF THE SHIP IS IN ORBIT AND RUNS OUT OF ELECTRICITY
			// WHILE THE DRIVE IS ACTIVE, IT WILL RE-CALCULATE THE ORBIT, WHICH COULD LEAD TO UNINTENDED SUBORBITAL TRAJECTORIES
			if (HighLogic.LoadedSceneIsFlight && driveActive && (ecAmnt < electricityReq))
			{
				// This will initiate the Disengage() function to determine an orbit, only if the ship is currently
				// travelling at warp velocities, if not, it will simply deactivate the drive.
				if (throttle > 0f)
				{
					Disengage ();
				}

				driveActive = false;
				this.Shutdown ();
				currentWF = 0f;
				ScreenMessages.PostScreenMessage ("WARP DRIVE DEACTIVATED - NOT ENOUGH ELECTRIC CHARGE!");
			}

            //*********************************************************************************************
            // Begin Non-Physical Time Warp code                 Added February 2024                      *
            //*********************************************************************************************

			// This conditional is entered only if Non-Physical Time Warp is entered.
			// Added February 2024
			if (HighLogic.LoadedSceneIsFlight && driveActive && ((TimeWarp.CurrentRate > 1f) && TimeWarp.WarpMode == TimeWarp.Modes.HIGH) && previousThrotState > 0f)
			{
				// These messages will be posted to the screen if Non-Physical Time Warp is engaged while
				// traveling at warp velocities.  The warp drive will disengage as it did before, but this message
				// will now see this message.  Physical Time Warp now yields upscaled velocities creating the
				// equivalent of travel at high levels of Non-Physical Time Warp while avoiding unplanned ship destruction
				// when exiting time warp, and allowing the warp star effect to continue to be seen.
				ScreenMessages.PostScreenMessage("WARP DRIVE DISENGAGED UNDER NON-PHYSICAL TIME WARP");
				ScreenMessages.PostScreenMessage("PLEASE USE PHYSICAL TIME WARP INSTEAD.");
            }
			//*************************************************
			// End Non-Physical Time Warp code                *
			//*************************************************

			if (HighLogic.LoadedSceneIsFlight && driveActive && throttle > 0f)
			{
				// Conditional to maintain the current warp factor if the selectedWarp is increased.
				// This is done by computing the ratio of lastSelectedWarp to selectedWarp and scaling the throttle appropriately
				// This conditional will only be true if the current selectedWarp is greater than lastSelectedWarp.
				// By doing this, increaing selectedWarp will not speed us up without user input, but decreasing it will slow us down.
				if (selectedWarp > lastSelectedWarp)
				{
					//throttle = (lastSelectedWarp / selectedWarp) * throttle;
					FlightInputHandler.state.mainThrottle = (lastSelectedWarp/selectedWarp) * throttle;
				}

				// Engage warp flight
				Engage(throttle);

				// Consume electric charge based on throttle setting
				this.vessel.RequestResource (this.part, ecID, electricityReq * (1 + throttle) * TimeWarp.fixedDeltaTime, true);

				// Check coil charge status if all coils are charged in this update
				if (allCoilsCharged)
				{
					numCoilsCharged = 0;
					CheckCharge();
				}

				// If the coils have been discharged, disengage the warp drive
				// and safely drop to impulse/conventional flight
				if (!allCoilsCharged && throttle > 0f)
				{
					Disengage ();
					driveActive = false;
					this.Shutdown ();
					currentWF = 0f;
					ScreenMessages.PostScreenMessage ("WARP COILS DISCHARGED - CANNOT GO TO WARP!");
				}

				previousThrotState = throttle;
			}

            //Debug.Log("[TrekDrive]: Throttle: " + throttle + " Previous Throttle State: " + previousThrotState + " Previous TimeWarp State: " + prevTimeWarpState);
            if (throttle == 0f && previousThrotState > 0f && !prevTimeWarpState)
			{
				Disengage ();
				previousThrotState = 0f;
				Debug.Log ("[TrekDrive] Drive Disengaged.");
			}

            //if (HighLogic.LoadedSceneIsFlight && !prevTimeWarpState && !timeWarpEnded)
            //{
            //    Debug.Log("[TrekDrive]: Made it here after timewarp ended, before next frame.");
            //}

            // Save the selectedWarp at the end of each FixedUpdate.
            // This will (hopefully) be used to scale the throttle so that if we increase the selectedWarp
            // The ship will not speed up without user input.  This may need to be removed, but we'll see.
            lastSelectedWarp = selectedWarp;
				
		}

		// The code that moves the ship through space at warp
		public void Engage(float throttleSetting)
		{
			// Variables necessary for warp travel to work
			// The Speed of Light, c
			// The initial speed of the vessel, to which we add our warp speed
			// The initial distance traveled
			int speedOfLight = 299792458;
			double speed = orbitalSpeed;
			double distance = 0d;

			// New variables to allow for Physical Time Warp to replace Non-Physical Time Warp
			// The timeWarpMult variable holds the current integer index of the time warp rate
			// The scaleFactor variable is used to compute the multiplying factor for the speed.
			// Added: 7 February 2024
			int timeWarpMult = TimeWarp.CurrentRateIndex;
			double scaleFactor = 1;

			// A swtich statement, inside a simple conditional, that selects the scaleFactor to be used
			// The conditional will only be entered if Physical Time Warp is being used.
			// This provides the equivalent of high levels of Non-Physical Time Warp without the detriments thereof.
			if (TimeWarp.WarpMode == TimeWarp.Modes.LOW)
			{
				switch(timeWarpMult)
				{
					case 3:
						scaleFactor = 10000;
						break;
					case 2:
						scaleFactor = 1000;
						break;
					case 1:
						scaleFactor = 100;
						break;
					case 0:
						scaleFactor = 1;
						break;
					default:
						scaleFactor = 1;
						break;
				}
			}

			// Compute the gravity breaking by calling the function
			GravityBrake ();

			// Compute the current magnitude of our velocity
			//speed += Math.Pow((maxWarp*throttleSetting*GravityBrakes),3)*speedOfLight;
			//speed += Math.Pow((maxWarp*throttleSetting*GravityBrakes),(0.25*maxWarp*throttleSetting + 2.5))*speedOfLight;			// This is the original that works properly, uncomment if the selectedWarp version does not work.
			speed += Math.Pow((selectedWarp*throttleSetting*GravityBrakes),(0.25*selectedWarp*throttleSetting + 2.5))*speedOfLight;

			// Compute the distance travelled since the last frame
			distance = scaleFactor * speed * TimeWarp.fixedDeltaTime;

			// The vector giving the new position based on our distance traveled this frame
			Vector3d position = ship.transform.position + (transform.up * (float)distance);

			// Translate the vessel through space.
			FloatingOrigin.SetOutOfFrameOffset (position);

			// Test Statements to determine the current Physical Time Warp level
			// as a replacement for Non-Physical Time Warp
			if (TimeWarp.WarpMode == TimeWarp.Modes.LOW)
			{
				Debug.Log("[TrekDrive]: Current Physical Time Warp Rate is " + TimeWarp.CurrentRateIndex);
			}
		}

		// The code that disengages the warp drive, returning the vessel to impulse/conventional propulsion
		public void Disengage()
		{
			// If the orbit mode is set to "Easy" (the default), we execute this code to set an orbit.
			// If the orbit mode is set to "Realistic", nothing is done.
			if (orbitMode)
			{
				CelestialBody curBody = ship.orbit.referenceBody;

				// Go on rails before setting the orbit
				ship.GoOnRails();

				// Get the position vector from the reference body, then set the velocity vector based on the ship's forward vector
				// with the magnitude set to that for a circular orbit at the current position
				Vector3d position = ship.orbit.pos.xzy;
				Vector3d velocity = Vector3d.Normalize(-1*ship.GetFwdVector())*Math.Sqrt (ship.orbit.referenceBody.gravParameter / Vector3d.Magnitude(position));

				// Update the orbti using the above state vectors
				ship.orbit.UpdateFromStateVectors(position, velocity, curBody, HighLogic.CurrentGame.UniversalTime);

				// Now, we can safely go off rails
				ship.GoOffRails();

			}

		}

		// Define a function "GravityBrake", heavily inspired by the USI Warp Drive function of the same name
		// This reduces the maximum warp when approaching a gravitational body.
		public void GravityBrake()
		{
			var cutoffRad = 0.95d * ship.mainBody.Radius;
			var curBody = ship.mainBody;
			GravityBrakes = (Math.Pow(cutoffRad,0.9d))/(Math.Pow(ship.orbit.radius,0.9d));
			if (curBody != FlightGlobals.Bodies.FirstOrDefault(x => x.referenceBody == x.referenceBody))
			{
				var nextBody = ship.mainBody.orbit.referenceBody;
				while (nextBody != FlightGlobals.Bodies.FirstOrDefault(x => x.referenceBody == x.referenceBody))
				{
					cutoffRad = 0.95d * nextBody.Radius;
					GravityBrakes += Math.Pow (cutoffRad, 0.9d) / Math.Pow (curBody.orbit.radius, 0.9d);
					curBody = nextBody;
					nextBody = nextBody.orbit.referenceBody;
				}
			}
			GravityBrakes = 1 - GravityBrakes;
		}

	}





	//###################################################################################################
	// Impulse Engine
	// Uses Liquid Deuterium to provide sub-light propulsion
	//###################################################################################################
	public class SW_ModuleImpulseEngine : ModuleEngines
	{
		[KSPField]
		public float maxThrust = 0f;

		[KSPField]
		public float maxAccel = 2.0f;			//This is in units of G

		[KSPField]
		public float maxVerticalAccel = 1.0f;	//This is in units of m/s^2

		// THINK ABOUT ADDING A SLIDER FOR VERTICAL ACCELERATION.
		// This would require an additional variable, verticalAccel, that would replace "maxVerticalAccel" in the
		// Takeoff, Hover, and Landing code.  verticalAccel would default to the maxVerticalAccel value, and the
		// maxVerticalAccel variable would be control the range of the slider, which would range from 0 to maxVerticalAccel.

		[KSPField]
		public float hoverAlt = 500f;			//This is in units of meters

		[KSPField]
		public float engineEfficiency = 0.75f;

		[KSPField]
		public float minDeuterium = 0.1f;

		[KSPField]
		public float ecProd = 1000;

		[KSPField]
		public float maxVectorAngle = 15f;		// This is in units of degrees

		// The path to the engine sound effect
		[KSPField]
		public string engineSFX = "TrekDrive/Sounds/tng_engine_hum_short";

		// Toggle for the hover mode in the PAW
		[KSPField(groupName = "ImpulsePAW", groupDisplayName = "Impulse Engine Operation", guiName = "Hover Mode", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Landing", enabledText = "Takeoff")]
		public bool takeOff = true;

		// The KSPAction associated with the hover mode.  This allows the hover mode to be a setable action group.
		[KSPAction(guiName = "Toggle Hover Mode", isPersistent = true)]
		public void ToggleModeAction(KSPActionParam param)
		{
			if (!takeOff)
			{
				takeOff = true;
			}
			else
			{
				takeOff = false;
			}
		}

		// Toggle for the translation mode in the PAW, either forward or reverse.  Default is "true", which is forward, "false" is reverse.
		[KSPField(groupName = "ImpulsePAW", guiName = "Translation Mode", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Reverse", enabledText = "Forward")]
		public bool thrustFwd = true;

		// The KSPAction associated with the translation mode
		[KSPAction(guiName = "Toggle Translation Mode", isPersistent = true)]
		public void ToggleTanslationAction(KSPActionParam param)
		{
			if (!thrustFwd)
			{
				thrustFwd = true;
			}
			else
			{
				thrustFwd = false;
			}
		}

		// Toggle for the Thrust Vectoring of the impulse engines.  If true, thrust vectoring is enabled.  Thrust vecoring is disabled if false.
		[KSPField(groupName = "ImpulsePAW", guiName = "Thrust Vectoring", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Disabled", enabledText = "Enabled")]
		public bool thrustVectoring = true;

		// The KSPAction associated with the thrust vectoring activation
		[KSPAction(guiName = "Toggle Thrust Vectoring", isPersistent = true)]
		public void ToggleVectoringAction(KSPActionParam param)
		{
			if (!thrustVectoring)
			{
				thrustVectoring = true;
			}
			else
			{
				thrustVectoring = false;
			}
		}

		// PAW variables to see if ctrlState.pitch/roll/yaw hold the inputs for the same from manual and SAS inputs
		// CONFIRMED - this commented out block can be safely removed at a later date.
		/*[KSPField(groupName = "ImpulsePAW", guiName = "Pitch", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public float currentPitchInput = 0f;

		[KSPField(groupName = "ImpulsePAW", guiName = "Roll", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public float currentRollInput = 0f;

		[KSPField(groupName = "ImpulsePAW", guiName = "Yaw", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public float currentYawInput = 0f;

		[KSPField(groupName = "ImpulsePAW", guiName = "Vessel Size", isPersistant = true, guiActiveEditor = false, guiActive = true)]
		public Vector3 sizeOfVessel;
		*/

		// Initialize the AudioSource for the engine sound effect
		private AudioSource engineSound;
		// Engine ignition boolean so that the sound starts once when the engine is activated
		// and stops only once the engine has been shutdown
		private bool isNewlyStarted;

		// Called when the part is started by Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);

			// Initialize the engine sound effect
			if (HighLogic.LoadedSceneIsFlight)
			{
				engineSound = this.part.gameObject.AddComponent<AudioSource>();
				engineSound.clip = GameDatabase.Instance.GetAudioClip (engineSFX);
				engineSound.volume = GameSettings.SHIP_VOLUME;
				//engineHum.audio.volume = 1.0f;

				engineSound.panStereo = 0;
				engineSound.rolloffMode = AudioRolloffMode.Linear;
				engineSound.loop = true;
				engineSound.spatialBlend = 1f;
			}
		}

		// Called when the part is made active
		public override void OnActive()
		{
			// This assumes that the part is made active when beginning flight.
			// Set takeOff to "true".
			// This shouldn't affect landing in an adverse manner.
			if (HighLogic.LoadedSceneIsFlight)
			{
				//takeOff = true;
				base.OnActive ();
				isNewlyStarted = true;
			}
		}

		// Define the throttle variable and a variable to hold the current active vessel, and the input values for pitch, roll, and yaw
		private float throttle;
		private Vessel ship;
		private float pitchInput;
		private float rollInput;
		private float yawInput;

		// Variables to control the thrust vectoring
		private float fwdThrust;
		private float pitchThrust;
		private float rollThrust;
		private float yawThrust;
		private Vector3 sizeOfVessel;

		// Deuterium related variables
		private double deutAmnt;
		private double deutMaxAmnt;

		// Get the resource ID for "LqdDeuterium" and for "ElectricCharge"
		public int deutID = PartResourceLibrary.Instance.GetDefinition("LqdDeuterium").id;
		public int ecID = PartResourceLibrary.Instance.GetDefinition("ElectricCharge").id;

		// Initialize the hover control time variable
		private float hoverTime = 0f;

		// Integer variable to determine the thrust direction
		private int thrustDirection;

		// The acceleration variables from the CFG are in Gs.  These variables will hold the actual acceleration numbers
		//private float maxAccelActual = maxAccel * 9.81f;
		//private float maxVertAccelActual = maxVerticalAccel * 9.81f;

		// This variable and function that holds and determines the number of active impulse engines, respectively, on the ship
		// This is necessary to ensure that the weight cancellation actually cancels the weight of the vessel
		// when there are multiple impulse engines on the vessel.
		private int numActiveEngines = 0;

		private void ActiveImpulse()
		{
			// Loop through all of the parts
			for (int i = 0; i < FlightGlobals.ActiveVessel.parts.Count; i++)
			{
				PartModuleList partModules = FlightGlobals.ActiveVessel.parts [i].Modules;
				for (int j = 0; j < partModules.Count; j++)
				{
					if (partModules[j] is SW_ModuleImpulseEngine && FlightGlobals.ActiveVessel.parts[i].FindModulesImplementing<SW_ModuleImpulseEngine>().First().EngineIgnited)
					{
						numActiveEngines += 1;
						//Debug.Log ("[TrekDrive] Number of Active Impulse Engines: " + numActiveEngines);
					}
				}
			}
		}

		// The FixedUpdate function, the meat of the module
		public void FixedUpdate()
		{

			// Set the "throttle" variable only if we are in flight, and get the connected resource totals for
			// electric charge only if we are in flight as well as the "ship" variable to hold the active vessel.
			if (HighLogic.LoadedSceneIsFlight)
			{
				// Set the "throttle" variable with the current mainThrottle percentage
				throttle = FlightGlobals.ActiveVessel.ctrlState.mainThrottle;

				// Set "ship" to the active vessel to simplify the overall code.
				//ship = FlightGlobals.ActiveVessel;
				ship = this.vessel;

				// Get the amount of liquid deuterium in the vessel.
				this.vessel.GetConnectedResourceTotals (deutID, out deutAmnt, out deutMaxAmnt, true);

				// Set the thrustDirection variable based on the selected mode.
				if (thrustFwd)
				{
					thrustDirection = 1;
				}
				else
				{
					thrustDirection = -1;
				}

				// Set the variables to hold the current input values for pitch, roll, and yaw
				pitchInput = FlightGlobals.ActiveVessel.ctrlState.pitch;
				rollInput = FlightGlobals.ActiveVessel.ctrlState.roll;
				yawInput = FlightGlobals.ActiveVessel.ctrlState.yaw;

				sizeOfVessel = ship.vesselSize;

				// Check for if the engine is shutdown.
				// If this is the case, stop playing the sound and set isNewlyStarted = true
				if (!this.EngineIgnited)
				{
					engineSound.Stop ();
					isNewlyStarted = true;
				}

			}

			// The below code needs to execute while the engine is active
			if (HighLogic.LoadedSceneIsFlight && this.EngineIgnited)
			{
				// Generate ElectricCharge to emulate the fusion reactor that is the core of the impulse drive.
				this.vessel.RequestResource (this.part, ecID, -ecProd * TimeWarp.fixedDeltaTime, true);

				// Update the propellant gauge
				this.UpdatePropellantGauge(this.propellants.FirstOrDefault());
				this.UpdatePropellantStatus (true);

				// Start playing the engine sound if the engine is newly activated (staged or activated after shutdown)
				if (isNewlyStarted)
				{
					engineSound.Play ();
					isNewlyStarted = false;
				}

				// Set the pitch of the sound based on throttle percentage
				engineSound.pitch = (float)Math.Pow(throttle,0.1);
			}


			// Takeoff and Landing code is encapsulated in this conditional.
			// This conditional will only evaluate true if the vessel is in a suborbital trajectory.
			// In this manner we prevent continual checks for takeoff and landing, and any unwanted forces while in an orbial situation
			if (HighLogic.LoadedSceneIsFlight && (ship.situation != Vessel.Situations.ORBITING || ship.situation != Vessel.Situations.ESCAPING || ship.situation != Vessel.Situations.DOCKED || ship.situation != Vessel.Situations.PRELAUNCH && ship.radarAltitude <= 2f*hoverAlt))
			{
				// Reset the number of active engines to 0, or else the number will grow and grow.
				numActiveEngines = 0;

				// Call the ActiveImpulse() function to determine how many impulse engines are active on the vessel
				ActiveImpulse();

				// Variable to control the takeoff thrust.
				// This variable will hold the predicted altitude at which the vessel will reach hoverAlt in 60sec
				// given the current vertical speed and radarAlt.  Once this value equals or exceeds hoverAlt
				// then the initial upward thrusting will stop and the vessel should coast to the hoverAlt
				// where it will not fall below while Takeoff is true.
				//float cutoffCondition;

				// Takeoff code
				if (this.EngineIgnited && (ship.radarAltitude < hoverAlt) && takeOff)
				{
					//Debug.Log ("[TrekDrive] Taking off"+Vector3.up);
					// This applies the force from the part on the ship's center-of-mass.
					// It works as expected, I just need to tweak the function to control the force applied.
					// I noted when testing this exact line that once it reaches the hover height, it begins to fall,
					// but the code is slowing it down until it started moving back upwards before striking the ground.
					//this.part.AddForceAtPosition ((-10f*(float)ship.totalMass * ship.graviticAcceleration.normalized), ship.CoM);

					//this.part.AddForceAtPosition (((-1f)*ship.graviticAcceleration.magnitude - 1f - (2f * ship.radarAltitude / hoverAlt))*(float)ship.totalMass*ship.graviticAcceleration.normalized, ship.CoM);

					// Calculate the cutoffCondtion using simple kinematics, ignoring air resistance.
					//cutoffCondition = (hoverAlt - (float)ship.radarAltitude) / ((float)ship.verticalSpeed/(2f*9.81f)) + (0.5f * 9.81f * ((float)ship.verticalSpeed/(2f*9.81f)));

					// If the vessel is below half of its hover altitude increase acceleration upward until that point.
					//if (ship.verticalSpeed < cutoffCondition)
					if (ship.radarAltitude/hoverAlt < 0.25f)
					{
						//this.part.AddForceAtPosition (((-1f) * ship.graviticAcceleration.magnitude - maxVerticalAccel)/numActiveEngines * (float)ship.totalMass * ship.graviticAcceleration.normalized, ship.CoM);

						// Replaced the original method of applying the force through the COM to applying the same acceleration to
						// each part with a rigid body.  This mitigates or eliminates the strange "slumping" of vessels as KSP
						// still keeps track of each part's mass, and does not treat the vessel as a monolithic object with a single mass.
						// This method is used in all of the Takeoff/Hover/Landing code.
						foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
						{
							// Skip parts with a physicalSignificance = NONE
							// This became necessary after discovering that a shuttlepod with ConformalDecals text decals
							// exhibited extreme acceleration in Takeoff mode (almost instantaeous shock effects at launch).
							// It seems that if the part has a rigidbody, but physical significance of "NONE", it applied the acceleration,
							// but as there was "no mass", it was applying multiples of the acceleration to the parent part (the Shuttlepod fuselage).
							if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
							{
								continue;
							}

							// Apply force in "Acceleration" mode, allowing me to just calculate the acceleration
							// and not have to worry about getting part dry mass and the mass of all resources.
							// This code is similarly applied (with appropriate modifications) in all of the Takeoff/Hover/Landing code.
							part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude - maxVerticalAccel)/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
						}

						//Debug.Log ("[TrekDrive] Current Thrust Percentage: " + this.thrustPercentage);
					}

					// If the vessel reaches half of its hover altitude simply switch to cancelling out gravity.
					// This should be sufficient time to slow down some to come to a hover, or simply continue upwards under impulse thrust
					if (ship.radarAltitude/hoverAlt >= 0.25f)
					{
						//this.part.AddForceAtPosition (((-1f)*ship.graviticAcceleration.magnitude)/numActiveEngines * (float)ship.totalMass*ship.graviticAcceleration.normalized, ship.CoM);
						foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
						{
							if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
							{
								continue;
							}
							part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude)/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
						}

						// Added to cancel out any negative vertical speed by adding maxVerticalAccel
						if (ship.verticalSpeed < 0f)
						{
							foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
							{
								if (part.physicalSignificance != Part.PhysicalSignificance.NONE)
								{
									part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude - maxVerticalAccel)/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
								}
							}
						}

					}

				}


				// Hover code - not a separate mode, just some code to get it to hover
				if (this.EngineIgnited && (ship.radarAltitude >= hoverAlt) && (ship.radarAltitude <= 2f*hoverAlt) && (throttle > 0f))
				{
					//this.part.AddForceAtPosition (((-1f)*ship.graviticAcceleration.magnitude)/numActiveEngines *(float)ship.totalMass*ship.graviticAcceleration.normalized, ship.CoM);
					foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
					{
						if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
						{
							continue;
						}
						part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude)/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
					}
				}



				// Landing code
				if (this.EngineIgnited && (ship.radarAltitude <= hoverAlt) && (ship.radarAltitude > 0d) && !takeOff)
				{
					//ship.acceleration = ship.graviticAcceleration - ship.graviticAcceleration - (2 * ship.radarAltitude * maxVerticalAccel / hoverAlt) * ship.up;
					//this.part.AddForceAtPosition(((-1)*ship.graviticAcceleration.magnitude - 0.001f + 0.5*maxVerticalAccel*Mathf.Sin(((float)ship.radarAltitude/hoverAlt)*Mathf.PI))*(float)ship.totalMass*ship.graviticAcceleration.normalized,ship.CoM);

					// If our vertical speed is positive, we should decrease any vertical forces to below the ship's weight
					if (ship.verticalSpeed > 0d)
					{
						//this.part.AddForceAtPosition (((-1f) * ship.graviticAcceleration.magnitude + maxVerticalAccel)/numActiveEngines * (float)ship.totalMass * ship.graviticAcceleration.normalized, ship.CoM);
						foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
						{
							if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
							{
								continue;
							}
							part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude + maxVerticalAccel)/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
						}
					}

					// If our vertical speed is negative, but above safe landing speed, apply a constant acceleration upwards
					if (ship.verticalSpeed < 0d && ship.verticalSpeed < -5d)
					{
						
						//this.part.AddForceAtPosition (((-1f) * ship.graviticAcceleration.magnitude - maxVerticalAccel*2f*(-1f*(float)ship.verticalSpeed/5f))/numActiveEngines * ship.totalMass * ship.graviticAcceleration.normalized, ship.CoM);
						foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
						{
							if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
							{
								continue;
							}
							part.Rigidbody.AddForce(((-1f) * ship.graviticAcceleration.magnitude - maxVerticalAccel*2f*(-1f*(float)ship.verticalSpeed/5f))/numActiveEngines * ship.graviticAcceleration.normalized,ForceMode.Acceleration);
						}
					}
				}
			}

			// Below is the code to control the actual thrusting of the impulse engines.  There is no need to account for the number
			// of active engines here as the maxAccel variable will be treated as the max acceleration provided by one engine, just
			// as one would expect of a normal ModuleEngines engine.
			/*if (HighLogic.LoadedSceneIsFlight && this.EngineIgnited && throttle > 0f && deutAmnt > minDeuterium)
			{
				// Testing the ctrlState pitch, roll, and yaw attributes to see if they capture the inputs
				//currentPitchInput = FlightGlobals.ActiveVessel.ctrlState.pitch;
				//currentRollInput = FlightGlobals.ActiveVessel.ctrlState.roll;
				//currentYawInput = FlightGlobals.ActiveVessel.ctrlState.yaw;

				// Drain Liquid Deuterium while the engine is thrusting
				this.vessel.RequestResource (this.part, deutID, minDeuterium*(1f-engineEfficiency) * (1 + throttle) * TimeWarp.fixedDeltaTime, true);

				// Actually apply the thrust to the ship center of mass along the forward vector
				this.part.AddForceAtPosition (maxAccel * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * thrustDirection * ship.vesselTransform.up,ship.CoM);

			}*/

			// Below is the updated code to control the primary thrusting capability of the impulse engines.  This version of the code,
			// the original of which is above commented out, adds "thrust vectoring" capability.  I have placed it in quotes because, in a way,
			// it is not really thrust vectoring, but applying off-axis acceleration to allow the impulse engines to participate in attitude
			// control.  This should emulate thrust vectoring and make the vessels easier to control in-atmosphere.
			if (HighLogic.LoadedSceneIsFlight && this.EngineIgnited && throttle > 0f && deutAmnt > minDeuterium)
			{
				// Testing the ctrlState pitch, roll, and yaw attributes to see if they capture the inputs
				//currentPitchInput = FlightGlobals.ActiveVessel.ctrlState.pitch;
				//currentRollInput = FlightGlobals.ActiveVessel.ctrlState.roll;
				//currentYawInput = FlightGlobals.ActiveVessel.ctrlState.yaw;

				// Drain Liquid Deuterium while the engine is thrusting
				this.vessel.RequestResource (this.part, deutID, minDeuterium*(1f-engineEfficiency) * (1 + throttle) * TimeWarp.fixedDeltaTime, true);

				// Set the angles to compute the components of the acceleration.
				// If thrust vectoring is enabled (thrustVectoring = true), add acceleration using components
				// If thrust vectoring is disabled (thrustVectoring = false), use the original behavior
				if (thrustVectoring)
				{
					// Apply acceleration by components
					// Need to combine the roll and yaw angles to get the off-axis component(s)
					// NOTE:  Pitch Up(S-key) is +1, Pitch Down(W-key) is -1, Roll Right(E-key) is +1, Roll Left (Q-key) is -1,
					// Yaw Right(D-key) is +1, and Yaw Left(A-key) is -1

					// Compute the component of forward thrust accounting for pitch, roll, and yaw
					fwdThrust = maxAccel * (float)Math.Cos(Math.PI*(double)(Math.Abs(pitchInput)*maxVectorAngle)/180) * (float)Math.Cos(Math.PI*(double)(Math.Abs(rollInput)*maxVectorAngle)/180) * (float)Math.Cos(Math.PI*(double)(Math.Abs(yawInput)*maxVectorAngle)/180);

					// Compute the component of pitch thrust
					pitchThrust = maxAccel * (float)Math.Sin(Math.PI*(double)(Math.Abs(pitchInput)*maxVectorAngle)/180);

					// Compute the component of roll thrust
					rollThrust = maxAccel * (float)Math.Sin(Math.PI*(double)(Math.Abs(rollInput)*maxVectorAngle)/180);

					// Compute the component of yaw thrust
					yawThrust = maxAccel * (float)Math.Sin(Math.PI*(double)(Math.Abs(yawInput)*maxVectorAngle)/180);

					// Apply the forward thrust
					// 6 Nov. 2021:  Changed ship.CoM to ship.CurrentCoM, this should prevent any future unexpected torques from changine CoM
					// Torques were noted on the Type F Shuttlecraft and the cause was found to be the production of warpPlasma below the CoM.  This production
					// caused the CoM to shift downward enough that the ship.CoM, which seems not to update each frame, was above the new actual CoM.
					// Hopefully using ship.CurrentCoM will prevent this and actually have thrust always applied through the CoM regardless of propellant usages,
					// or shuttlecraft embarked.
					//this.part.AddForceAtPosition(fwdThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * thrustDirection * ship.vesselTransform.up,ship.CoM);

					foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
					{
						if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
						{
							continue;
						}
						part.Rigidbody.AddForce(fwdThrust * 9.81f * (this.thrustPercentage/100f) * throttle * thrustDirection * ship.vesselTransform.up,ForceMode.Acceleration);
					}

					// Apply pitch thrust
					// If the pitch is a pitch up, the pitch input will be positive.  In this case, we apply the force (hopefully) in the forward direction (downward with respect
					// to the way the vessel actually flies) at a point along the long axis of the ship near the rear of the ship.
					// If the pitch is a pitch down, we simply flip the direction of the force by multiplying by -1.
					if (pitchInput > 0)
					{
						this.part.AddForceAtPosition(pitchThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * ship.vesselTransform.forward,(ship.CoM-(sizeOfVessel.z*ship.vesselTransform.up/2)));
					}
					else
					{
						this.part.AddForceAtPosition(pitchThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * (-1f) * ship.vesselTransform.forward,(ship.CoM-(sizeOfVessel.z*ship.vesselTransform.up/2)));
					}

					// Apply roll thrust
					// If the roll is to the right, the roll input will be positive.  In this case, we apply the force (hopefully) in the forward direction (downward with respect
					// to the way the vessel actually flies) at a point along the x-axis of the ship near the starboard side of the ship.
					// If the roll is to the left, we simply flip the direction of the force by multiplying by -1.
					if (rollInput > 0)
					{
						this.part.AddForceAtPosition(rollThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * ship.vesselTransform.forward,(ship.CoM+(sizeOfVessel.y*ship.vesselTransform.right/2)));
					}
					else
					{
						this.part.AddForceAtPosition(rollThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * (-1f) * ship.vesselTransform.forward,(ship.CoM+(sizeOfVessel.y*ship.vesselTransform.right/2)));
					}

					// Apply yaw thrust
					// If the yaw is to the right, the yaw input will be positive.  In this case, we apply the force (hopefully) in the port direction (to the left with respect
					// to the way the vessel actually flies) at a point along the long axis of the ship near the rear of the ship.
					// If the yaw is to the right, we simply flip the direction of the force by multiplying by removing the mulitplication by -1.
					if (yawInput > 0)
					{
						this.part.AddForceAtPosition(yawThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * (-1f) * ship.vesselTransform.right,(ship.CoM-(sizeOfVessel.z*ship.vesselTransform.up/2)));
					}
					else
					{
						this.part.AddForceAtPosition(yawThrust * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * ship.vesselTransform.right,(ship.CoM-(sizeOfVessel.z*ship.vesselTransform.up/2)));
					}

				}
				else
				{
					// Actually apply the thrust to the ship center of mass along the forward vector
					//this.part.AddForceAtPosition (maxAccel * 9.81f * (this.thrustPercentage/100f) * throttle * (float)ship.totalMass * thrustDirection * ship.vesselTransform.up,ship.CoM);

					foreach (var part in ship.parts.Where(p => p.Rigidbody != null))
					{
						if (part.physicalSignificance == Part.PhysicalSignificance.NONE)
						{
							continue;
						}
						part.Rigidbody.AddForce(maxAccel * 9.81f * (this.thrustPercentage/100f) * throttle * thrustDirection * ship.vesselTransform.up,ForceMode.Acceleration);
					}
				}

			}
		}
	}





	//###################################################################################################
	// Bussard Collector
	// Uses Electric Charge to collect deuterium from the surrounding environment.
	//###################################################################################################
	public class SW_ModuleBussardCollector : PartModule
	{
		[KSPField]
		public float collectionRadius = 10f;

		[KSPField]
		public float collectorEfficiency = 0.75f;

		// Toggle for the hover mode in the PAW
		[KSPField(groupName = "BussardPAW", groupDisplayName = "Bussard Collector", guiName = "Collector Status", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Inactive", enabledText = "Collecting")]
		public bool collecting = false;

		// The KSPAction associated with the collection state of the bussard.  This allows the collection state to be a setable action group.
		[KSPAction(guiName = "Toggle Bussard Collector", isPersistent = true)]
		public void ToggleCollectionAction(KSPActionParam param)
		{
			if (!collecting)
			{
				collecting = true;
			}
			else
			{
				collecting = false;
			}
		}


		// Variable initialization for the bussard collector
		private float elapsedTime = 0f;
		private float interval = 60f;
		private float deutDensity = UnityEngine.Random.Range(0f, 1f);
		private float crossSecArea;

		// Get the resource ID for "LqdDeuterium" and for "ElectricCharge"
		public int deutID = PartResourceLibrary.Instance.GetDefinition("LqdDeuterium").id;
		public int antiMatID = PartResourceLibrary.Instance.GetDefinition("Antimatter").id;

		private Vessel ship;

		// Called when the part is started by Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);
			crossSecArea = Mathf.PI * collectionRadius * collectionRadius;
		}


		// DriveActive function:
		// Checks to see if there are any active warp field generators (SW_ModuleWarpGenerator)
		// This is used to determine if it the bussard will collect particles, as I want to disallow collection at warp.
		// This makes sense as the nav deflector would deflect any particles from the ship's path.
		private bool DriveActive()
		{
			if (HighLogic.LoadedSceneIsEditor)
			{
				return false;
			}

			bool isDriveActive = false;
			for (int i = 0; i < FlightGlobals.ActiveVessel.parts.Count; i++)
			{
				PartModuleList partModules = FlightGlobals.ActiveVessel.parts [i].Modules;
				for (int j = 0; j < partModules.Count; j++)
				{
					if (partModules[j] is SW_ModuleWarpGenerator && FlightGlobals.ActiveVessel.parts[i] != this.part)
					{
						isDriveActive = FlightGlobals.ActiveVessel.parts [i].FindModulesImplementing<SW_ModuleWarpGenerator> ().First ().driveActive;
					}

					if (isDriveActive)
					{
						return true;
					}
				}
			}

			return false;
		}


		// Called each frame, the meat of the module
		public void FixedUpdate()
		{
			// The encapsulating conditional so that the code is only triggered in the Flight scene.
			// This should make things a little neater too.
			if (HighLogic.LoadedSceneIsFlight)
			{
				// Set the "ship" variable to the active vessel
				ship = FlightGlobals.ActiveVessel;

				// An encapsulating conditional to make certain that the ship is not landed, pre-launch, or crashed, and that the collector is on.
				// If that is true, then the code to determine the amount of deuterium collected will execute.
				if (ship.situation != Vessel.Situations.LANDED && ship.situation != Vessel.Situations.PRELAUNCH && collecting && !DriveActive() || (DriveActive() && FlightGlobals.ActiveVessel.ctrlState.mainThrottle == 0f))
				{
					// If the interval has not elapsed collect deuterium at the given concentration
					if (elapsedTime <= interval)
					{
						// Generate Liquid Deuterium based on the deuterium density, cross-sectional area, and collection efficiency
						this.vessel.RequestResource (this.part, deutID, -deutDensity * crossSecArea * collectorEfficiency/9020f, true);

						// Generate Antimatter based on the same as the deuterium, but scaled down significantly
						this.vessel.RequestResource(this.part, antiMatID, -deutDensity * crossSecArea * collectorEfficiency * 0.00001f, true);

						// Increment the elapsedTime by the fixedDeltaTime
						elapsedTime += TimeWarp.fixedDeltaTime;
					}

					// If the elapsed time is greater than the interval, set elapsedTime to 0, and choose a new interval and deutDensity
					if (elapsedTime > interval)
					{
						elapsedTime = 0f;
						interval = UnityEngine.Random.Range (0f, 180f);
						deutDensity = UnityEngine.Random.Range (0f, 1f);
					}
				}
			}
		}
	}


    //###################################################################################################
    // Bridge Ambience
    // For immersion, plays a looping bridge ambience sound while in IVA.
    //###################################################################################################
    public class SW_BridgeAmbience : PartModule
	{
		// The path to the ambience sound effect
		[KSPField]
		public string bridgeSoundPath = "TrekDrive/Sounds/ENT_BridgeAmbience";

		/*// Toggle button to power up the bridge (requires no resources).
		[KSPField(groupName = "CorePAW", groupDisplayName = "Bridge Operation", guiName = "Bridge Status", isPersistant = true, guiActive = true), UI_Toggle(disabledText = "Powered Down", enabledText = "Powered Up")]
		protected bool activateBridge = false;

		// KSPAction to allow the warp core to be toggle through an action group
		[KSPAction(guiName = "Toggle Bridge Ops",isPersistent = true)]
		public void ToggleBridgeOps(KSPActionParam param)
		{
			if (!activateBridge)
			{
				activateBridge = true;
			}
			else
			{
				activateBridge = false;
			}
		}*/


		// Variables to initialize and handle playing the sound
		// Variable to control the sound
		private AudioSource bridgeAmb;
		// Variable to control the sound playback
		private bool playSound = true;

		// Initialize the part when the part is started in Unity
		public override void OnStart(StartState state)
		{
			base.OnStart (state);

			// Initialize the sound gameObject
			if (HighLogic.LoadedSceneIsFlight)
			{
				bridgeAmb = this.part.gameObject.AddComponent<AudioSource>();
				bridgeAmb.clip = GameDatabase.Instance.GetAudioClip (bridgeSoundPath);
				bridgeAmb.volume = GameSettings.SHIP_VOLUME;
				//bridgeAmb.volume = 0f;
				//bridgeAmb.bypassListenerEffects = true;
				//engineHum.audio.volume = 1.0f;

				bridgeAmb.panStereo = 0;
				bridgeAmb.rolloffMode = AudioRolloffMode.Linear;
				bridgeAmb.loop = true;
				bridgeAmb.spatialBlend = 1f;
			}

		}


		public void FixedUpdate()
		{
			
			if (HighLogic.LoadedSceneIsFlight)
			{
				if (InternalCamera.Instance.isActive)
				{
					//bridgeAmb.volume = GameSettings.SHIP_VOLUME;
					if (playSound)
					{
						bridgeAmb.Play ();
						Debug.Log ("[TrekDrive] Bridge Ambient sound started.");
						playSound = false;
					}
				}
				else
				{
					//bridgeAmb.volume = 0f;
					if (!playSound)
					{
						bridgeAmb.Stop ();
						Debug.Log ("[TrekDrive] BridgeAmbient sound stopped.");
						playSound = true;
					}

				}
			}
			else
			{
				//bridgeAmb.Stop ();
				//bridgeAmb.volume = 0f;
				playSound = true;
			}
			/*if (HighLogic.LoadedSceneIsFlight)
			{
				if (InternalCamera.Instance.isActive)
				{
					bridgeAmb.volume = GameSettings.SHIP_VOLUME;
				}
				else
				{
					bridgeAmb.volume = 0f;
				}
			}

			if (activateBridge)
			{
				if (playSound)
				{
					bridgeAmb.Play ();
					playSound = false;
				}
			}
			else
			{
				if (!playSound)
				{
					bridgeAmb.Stop ();
					playSound = true;
				}
			}*/
		}
	}

}

