"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.downloadElectron = downloadElectron;

function _bluebirdLst() {
  const data = require("bluebird-lst");

  _bluebirdLst = function () {
    return data;
  };

  return data;
}

function _binDownload() {
  const data = require("builder-util/out/binDownload");

  _binDownload = function () {
    return data;
  };

  return data;
}

function _fsExtraP() {
  const data = require("fs-extra-p");

  _fsExtraP = function () {
    return data;
  };

  return data;
}

function os() {
  const data = _interopRequireWildcard(require("os"));

  os = function () {
    return data;
  };

  return data;
}

var path = _interopRequireWildcard(require("path"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

const debug = require("debug")("electron-download");

const envPaths = require("env-paths");

const ChecksumValidator = require("sumchecker").ChecksumValidator;

let tmpFileCounter = 0; // noinspection SpellCheckingInspection

const checksumFilename = "SHASUMS256.txt";

class ElectronDownloader {
  constructor(options) {
    if (!options.version) {
      throw new Error("version not specified");
    } // use passed argument or XDG environment variable fallback to OS default


    this.options = Object.assign({
      platform: os().platform(),
      arch: os().arch()
    }, options);
    this.options.isVerifyChecksum = this.options.isVerifyChecksum !== false;
    this.cacheDir = this.options.cache || process.env.ELECTRON_CACHE || envPaths("electron", {
      suffix: ""
    }).cache;
  }

  get baseUrl() {
    return process.env.NPM_CONFIG_ELECTRON_MIRROR || process.env.npm_config_electron_mirror || process.env.ELECTRON_MIRROR || this.options.mirror || "https://github.com/electron/electron/releases/download/v";
  }

  get middleUrl() {
    return process.env.ELECTRON_CUSTOM_DIR || this.options.customDir || this.options.version;
  }

  get urlSuffix() {
    return process.env.ELECTRON_CUSTOM_FILENAME || this.options.customFilename || this.filename;
  }

  get cachedChecksum() {
    return path.join(this.cacheDir, `${checksumFilename}-${this.options.version}`);
  }

  get checksumUrl() {
    return `${this.baseUrl}${this.middleUrl}/${checksumFilename}`;
  }

  get filename() {
    const type = `${this.options.platform}-${this.options.arch}`;
    const suffix = `v${this.options.version}-${type}`;
    return `electron-${suffix}.zip`;
  }

  verifyFile(cachedFile, removeIfInvalid) {
    var _this = this;

    return (0, _bluebirdLst().coroutine)(function* () {
      if (!_this.options.isVerifyChecksum) {
        return true;
      }

      debug("Verifying file with checksum");

      if (!(yield (0, _fsExtraP().pathExists)(_this.cachedChecksum))) {
        yield _this.downloadFile(_this.checksumUrl, _this.cachedChecksum);
      }

      const checker = new ChecksumValidator("sha256", _this.cachedChecksum);

      try {
        yield checker.validate(path.dirname(cachedFile), path.basename(cachedFile));
      } catch (e) {
        if (!removeIfInvalid) {
          throw e;
        }

        try {
          yield (0, _fsExtraP().unlink)(cachedFile);
        } catch (ignore) {// ignore
        }

        return false;
      }

      return true;
    })();
  }

  downloadFile(url, destination) {
    var _this2 = this;

    return (0, _bluebirdLst().coroutine)(function* () {
      const tempFileName = `tmp-${process.pid}-${(tmpFileCounter++).toString(16)}-${path.basename(destination)}`;
      const cacheDir = _this2.cacheDir;
      debug("downloading", url, "to", cacheDir);
      const tempFile = path.join(cacheDir, tempFileName);
      yield (0, _binDownload().download)(url, tempFile);
      debug("moving", tempFile, "from", cacheDir, "to", destination);

      try {
        yield (0, _fsExtraP().rename)(tempFile, destination);
      } catch (e) {
        try {
          yield (0, _fsExtraP().unlink)(tempFile);
        } catch (cleanupError) {
          try {
            console.error(`Error deleting cache file: ${cleanupError.message}`);
          } catch (ignore) {// ignore
          }

          throw e;
        }
      }
    })();
  }

  get cachedZip() {
    return path.join(this.cacheDir, this.options.customFilename || this.filename);
  }

  downloadIfNotCached() {
    var _this3 = this;

    return (0, _bluebirdLst().coroutine)(function* () {
      const url = `${_this3.baseUrl}${_this3.middleUrl}/${_this3.urlSuffix}`;
      let cachedFile = _this3.cachedZip;
      debug("info", {
        cachedFile,
        url
      });
      const exists = yield (0, _fsExtraP().pathExists)(cachedFile);

      if (exists) {
        debug("file exists", cachedFile);

        if (yield _this3.verifyFile(cachedFile, true)) {
          return cachedFile;
        }
      }

      debug("creating cache dir");

      try {
        yield (0, _fsExtraP().ensureDir)(_this3.cacheDir);
      } catch (e) {
        if (e.code !== "EACCES") {
          throw e;
        } // try local folder if homedir is off limits (e.g. some linuxes return '/' as homedir)


        const localCache = path.resolve("./.electron");
        yield (0, _fsExtraP().ensureDir)(localCache);
        _this3.cacheDir = localCache;
        cachedFile = _this3.cachedZip;
      }

      yield _this3.downloadFile(url, cachedFile);
      yield _this3.verifyFile(cachedFile, false);
      return cachedFile;
    })();
  }

}

function downloadElectron(options) {
  return new ElectronDownloader(options).downloadIfNotCached();
} 
//# sourceMappingURL=electron-download.js.map